program psk31;           {psk31(evm) driver   QPSK version}
{extended character-set version}
{$R+}
{

 Please contact the author peter.martinez@btinternet.com to report
 bugs or suggest changes. This is freeware for amateur use only

data format from evm to PC
fend,00,PPPPPPPP,XV0AAAAA  P=phase,X=xon, V=overrange, A=loglevel
fend,01,(64 bytes)   waterfall data. 500Hz wide. 96dB log
 'fend' in data replaced by 'fesc,tfend'
 'fesc' in data replaced by 'fesc,tfesc'

data format from PC to evm
 00DDDDDD       shift DDDDDD left into lsb-end of 24-bit stack
 0100TCID       data for transmission. T=txon
                C=convoluted, I=inverted, D=data
 1000LGTR       Copy stack to Level, Gain, Txfreq, Rxfreq
 11XXXXXX       not used

}

uses dos,crt,graph,mouse,evm;
const portadr:array[1..4] of word=($3F8,$2F8,$3E8,$2E8);
      picmask:array[1..4] of byte=($10,$08,$10,$08);
      intvec:array[1..4] of byte=(12,11,12,11);
      tbufsize=256;      {bits from characters going to EVM}
      kbufsize=2048;     {keypresses going to transmit characters}
      rbufsize=256;     {bits from EVM going to decoder}
      persist=8;        {tuning display persistence}
      samplerate=8000;   {evm sample rate for computing frequencies}
      dcdlimit=18;       {number of bits to test for carrier on/off detect}
      onoff:array[0..1] of string[3]=('OFF','ON');
      scopeX=74;
      scopeY=39;       {position of scope on screen}
      wfallX=150;
      wfallY=38;       {position of waterfall on screen}
      poly1=$19;
      poly2=$17;       {convolutional encoder polynomials}
      depth=20;        {Viterbi decoding delay (=4*k) }
      fend=$C0;
      fesc=$DB;
      tfend=$DC;
      tfesc=$DD;
      keytab:array[128..255] of byte=     {keycodes to ANSI}
     (199,252,233,226,228,224,229,231,234,235,232,239,238,236,196,197,
      201,230,198,244,246,242,251,249,255,214,220,162,163,165,000,000,
      225,237,243,250,241,209,170,186,191,174,172,189,188,161,171,187,
      000,000,000,000,000,193,194,192,169,000,000,000,000,162,165,000,
      000,000,000,000,000,000,227,195,000,000,000,000,000,000,000,164,
      240,208,202,203,200,205,207,000,000,000,000,000,000,000,204,000,
      211,223,212,210,245,213,181,254,222,218,219,217,253,221,000,000,
      000,177,000,190,182,167,247,000,176,168,000,000,179,178,000,000);
      scrtab:array[128..255] of byte=     {ANSI to screencodes}
     (000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,
      000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,
      000,173,155,156,207,190,000,245,000,184,166,174,170,000,169,000,
      248,241,253,252,000,230,244,250,000,251,167,175,172,171,243,168,
      000,181,182,199,142,143,146,128,212,144,210,211,000,214,215,216,
      209,165,227,224,226,229,153,000,157,235,233,234,154,237,232,225,
      133,160,131,198,132,134,145,135,138,130,136,137,141,161,140,139,
      208,164,149,162,147,228,148,246,000,151,163,150,129,236,231,152);

type keyproc=procedure(C:char);
     Tcode=(off,tune,bpsk,qpsk,morse);        {operating modes}
     Tplot=record                  {tuning display position}
      X,Y:word
     end;
     Tbuf=record                   {buffer structure}
           data:array[0..tbufsize-1] of byte;
           inptr,outptr:word
          end;
     Kbuf=record
           data:array[0..kbufsize-1] of char;
           inptr,outptr:word
          end;
     Rbuf=record
           data:array[0..rbufsize-1] of byte;
           inptr,outptr:word
          end;
     Tstate=record        {a state in the trellis}
             dist:byte;     {total excess distance to this state}
             esti:longint;  {estimate of transmitted bit sequence}
            end;

var psk31dir:dirstr;
    callsign:string[10];
    cwid:string[20];
    comportNr:byte;
    error:byte;    {used to look at evm loader errors}
    rxbuf:Rbuf;
    txbuf:Tbuf;
    keybuf:Kbuf;    {buffers for stuff to/from evm}
    existcom:pointer; {com port interrupt vector to go back afterwards}
    varsreg:word;                     {for decoding varicode}
    rxdata,bitcount,column,truecount,falsecount:byte;
    init,boot,afc,neton,rev,autolf,soundon,fileout,Xonoff,overrange,abort:boolean;
    simplex,logon,txon,quit,toolong,dcd,esc:boolean;
    rxfreq,txfreq:real;                    {frequencies in Hz}
    rxfstr:string[6];                      {rxfreq as shown on screen}
    outfile,logfile:text;
    filename:string[63];
    keydo:keyproc;
    code:Tcode;
    encotab:array[0..255] of word;      {encode table for varicode}
    decotab:array[0..2047] of char;     {decode table for varicode}
    echoline:string[42];                {echoes keyboard input}
    scope:array[0..persist-1] of Tplot;  {stores tuning display history}
    tuneptr:byte;                       {points to tuning display history}
    convshift:byte;                     {convolutional code shift register}
    lasttx:byte;                        {last transmit phase value}
    symbols:array[0..31] of byte;       {convolutional code symbols}
    trellis:array[0..15] of Tstate;     {viterbi decoder trellis}
    rxstate:byte;
    rxframe:array[0..64] of byte;
    wbuf:array[0..2047] of byte;        {waterfall image buffer}

{$L egavga.obj}
procedure egavga; external;     {dummy procedure for VGA driver}

procedure startgraph;
var GrDr,GrMo:integer;
begin
 GrDr:=registerbgidriver(@egavga);
 GrDr:= detect;
 initgraph(GrDr,GrMo,'');
 if graphresult<>0 then halt;
end;

function nextT(ptr:word):word;    {move buffer pointer around circular buffer}
begin
 nextT:=(ptr+1) mod Tbufsize
end;

function nextk(ptr:word):word;
begin
 nextk:=(ptr+1) mod kbufsize
end;

function nextr(ptr:word):word;
begin
 nextr:=(ptr+1) mod rbufsize
end;

procedure initvari(Fname:string);   {initialise varicode lookup tables}
var F:text;
    I:integer;
    W,mask,R:word;
    S:string[12];
begin
 assign(F,Fname);
 {$I-}
 reset(F);
 readln(F,S);
 for I:=0 to 2047 do decotab[I]:=#0;   {prefill with error symbol NOT NEEDED}
 I:=0;
 while not(eof(F)) do
 begin
  readln(F,S);                          {get varicode string from file}
  W:=4;
  R:=1;
  while S<>'' do                        {copy string chars into codeword bits}
  begin
   W:=W shl 1;
   R:=R shr 1;
   if S[length(S)]='1' then
   begin
    inc(W);
    R:=R or $8000
   end;
   S:=copy(S,1,length(S)-1)
  end;
  encotab[I]:=W;                          {put codeword into encode table}
  while not(odd(R)) do R:=R shr 1;        {shift to lsb=1}
  R:=R shr 1;                             {shift the end '1' out}
  decotab[R]:=char(I);            {fill in decode table entry}
  inc(I)
 end;
 close(F);
 varsreg:=0;
 toolong:=false
 {$I+}
end;

function parity(data:byte):byte;
var count:byte;
begin
 count:=0;
 while (data>0) do
 begin
  if odd(data) then inc(count);
  data:=data shr 1
 end;
 parity:=count and 1
end;

procedure setafc(A:boolean);
begin
 setcolor(black);
 mouseoff;
 if init then outtextXY(565,5,onoff[byte(afc)]);
 afc:=A;
 if afc then setcolor(lightgreen) else setcolor(lightred);
 outtextXY(565,5,onoff[byte(afc)]);
 mouseon
end;
                              
procedure setdcd(D:boolean);
begin
 setcolor(black);
 mouseoff;
 if init then outtextXY(455,5,onoff[byte(dcd)]);
 dcd:=D;
 if dcd then setcolor(lightgreen) else setcolor(lightred);
 outtextXY(455,5,onoff[byte(dcd)]);
 mouseon
end;

procedure setfile(S:boolean);
begin
 fileout:=S;
 mouseoff;
 if fileout then setcolor(lightgreen) else setcolor(black);
 outtextXY(350,50,'FILE SEND');
 mouseon
end;

procedure setlog(L:boolean);
begin
 setcolor(black);
 mouseoff;
 if init then outtextXY(455,35,onoff[byte(logon)]) else logon:=false;
 if L then
 begin
  assign(logfile,psk31dir+'PSK31.LOG');
  {$I-}
  append(logfile);
  {$I+}
  if ioresult>0 then rewrite(logfile)
 end else
 begin
  if logon then close(logfile)
 end;
 logon:=L;
 if logon then setcolor(lightgreen) else setcolor(lightred);
 outtextXY(455,35,onoff[byte(logon)]);
 mouseon
end;

procedure scroll;
var R:registers;
begin
 R.ax:=$0601;     {scroll one "line" of 16 pixels}
 R.bx:=$0000;     {black}
 R.ch:=0;         {from top}
 R.cl:=0;         {from left}
 R.dh:=24;        {to bottom}
 R.dl:=79;        {to right}
 intr($10,R);     {get BIOS to do it}
end;

procedure linefeed;
begin
 if whereY=25 then scroll else write(#10)
end;

procedure writescreen(C:char);
begin
 if logon then
 begin
 {$I-}
  if (C<>#10) or not(autolf) then write(logfile,C);
  if (C=#13) and autolf then write(logfile,#10);
 {$I+}
  if ioresult>0 then setlog(false)
 end;
 case C of
   #8: write(#8' '#8);
  #10: if not(autolf) then linefeed;
  #13: begin
       write(#13);
       if autolf then linefeed;
      end;
   #32..#255:
     begin
      if whereX=80 then
      begin
       write(#13);
       if autolf then linefeed;
      end;
      write(C);
     end
 end
end;

procedure writerx(C:char);
begin
 if C>#127 then C:=char(scrtab[byte(C)]);    {convert ANSI to CP850}
 if not(simplex and txon) and (C>#0) then writescreen(C)
end;

procedure primetrellis; {set trellis as if it had been receiving 1's for ever}
var I:integer;
const dprime:array[0..15] of byte=(9,10,5,7,7,5,8,9,7,5,7,6,5,4,2,0);
begin
 for I:=0 to 15 do trellis[I].dist:=dprime[I];
 for I:=0 to 15 do trellis[I].esti:=$FFFFFFF0+I
end;

procedure dodcd(DP:byte);
var carrier,reversals:boolean;
    I:integer;
begin
 carrier:=(((DP+$20) shr 6) and 3)=0;
 reversals:=(((DP+$20) shr 6) and 3)=2;
 if carrier then
 begin
  truecount:=0;
  if falsecount<dcdlimit then
  begin
   inc(falsecount);
   if falsecount=dcdlimit then
   begin
    setdcd(false);
    varsreg:=0;              {prime decoders after dropout}
    toolong:=false;
    primetrellis;
   end
  end
 end else
 if reversals then
 begin
  falsecount:=0;
  if truecount<dcdlimit then
  begin
   inc(truecount);
   if truecount=dcdlimit then setdcd(true)
  end
 end else
 begin
  truecount:=0;
  falsecount:=0
 end
end;

procedure decodeB(bit:boolean);
begin
 varsreg:=varsreg shl 1;                     {shift previous bits}
 if varsreg and $C000>0 then toolong:=true;  {mark codes length>12}
 if bit then varsreg:=varsreg or 1;      {add in this bit}
 if (varsreg and 7)=4 then      {if end of code}
 begin
  varsreg:=varsreg shr 3;  {remove gap and first '1'}
  if not(toolong) and (varsreg<=encotab[$FF]) then writerx(decotab[varsreg]);  {lookup and out}
  toolong:=false;                                            {reset flag}
  varsreg:=0                                                 {reset shiftreg}
 end
end;                                       

function distance(A,B:byte):byte;         {distance=approx euclidean}
const dlookup:array[0..3,0..3] of byte=
      ((0,2,3,2),(2,0,2,3),(3,2,0,2),(2,3,2,0));
begin
 distance:=dlookup[A,B]
end;

procedure decodeV(rcvd:byte);    {The Viterbi Decoder}
var I,J:integer;
    dists:array[0..31] of byte;
    ests:array[0..31] of longint;
    select,min,vote:byte;
begin
 if rev then rcvd:=(4-rcvd) and 3;
 min:=255;
 for I:=0 to 31 do      {calc distances for all states and both data values}
 begin                 {added distance=distance between rcvd and predicted symbol}
  dists[I]:=trellis[I div 2].dist+distance(rcvd,symbols[I]);
  if dists[I]<min then min:=dists[I];    {keep track of the smallest distance}
  ests[I]:=((trellis[I div 2].esti) shl 1)+(I and 1)  {new estimate}
 end;
 for I:=0 to 15 do         {for each new state in the new trellis array}
 begin
  if dists[I]<dists[16+I] then select:=0 else select:=16;  {select lowest}
  trellis[I].dist:=dists[select+I]-min;          {update excess distances}
  trellis[I].esti:=ests[select+I]                {keep the new estimate}
 end;
 vote:=0;
 for I:=0 to 15 do if (trellis[I].esti and (1 shl depth))>0 then inc(vote);
 if vote=8 then decodeB(random(2)=1) else decodeB(vote<8);
end;

procedure initviterbi;
var I:integer;
    B:byte;
begin
 for I:=0 to 31 do                       {..generate the encode table}
  symbols[I]:=2*parity(I and poly1)+parity(I and poly2);
 primetrellis;
 convshift:=$FF             {initialise transmit shiftregister for idles}
end;

procedure decode(DP:byte);
begin
 dodcd(DP);
 if dcd then
 case code of
  qpsk:     decodeV(((DP+$20) shr 6) and 3);        {phase shift 0..3}
  bpsk: decodeB((DP+$40) and $FF<$80);              {reversal=logic 0}
 end
end;

procedure showwaterfall;
type Tscale=array[0..15] of byte;
const scale:Tscale=(0,1,8,4,5,9,6,2,3,12,7,13,10,11,14,15);
var I,A,B,C:integer;
begin
 mouseoff;
 getimage(wfallX,wfallY,wfallX+95,wfallY+10,wbuf);
 putimage(wfallX,wfallY+1,wbuf,normalput);
 if overrange then
 begin
  setcolor(white);
  setlinestyle(solidln,0,1);
  line(wfallX,wfallY,wfallX+95,wfallY);
  overrange:=false;
  mouseon;
  exit
 end;
 A:=rxframe[0];
 rxframe[64]:=rxframe[63];
 for I:=0 to 31 do
 begin
  B:=rxframe[2*I+1];
  C:=(A+3*B) div 64;
  putpixel(wfallX+3*I,wfallY,scale[A div 16]);
  putpixel(wfallX+3*I+1,wfallY,scale[C]);
  A:=rxframe[2*I+2];
  C:=(3*A+B) div 64;
  putpixel(wfallX+3*I+2,wfallY,scale[C])
 end;
 mouseon;
end;

procedure showscope(dph,amp:byte);
var I:integer;
begin
 mouseoff;
 setcolor(black);
 setlinestyle(solidln,0,3);
 line(scope[tuneptr].X,scope[tuneptr].Y,scopeX,scopeY);  {black out ancient line}
 scope[tuneptr].X:=scopeX+round(amp*sin(pi*dph/128));
 scope[tuneptr].Y:=scopeY-round(amp*cos(pi*dph/128))+1;  {calc coords of new line}
 if dcd then setcolor(yellow) else setcolor(red);
 line(scope[tuneptr].X,scope[tuneptr].Y,scopeX,scopeY);    {paint new line}
 tuneptr:=(tuneptr+1) mod persist;               {count round persistence buffer}
 mouseon
end;

procedure doafc(dph:byte);
var pp:integer;
begin
 if code=morse then exit;
 if code=bpsk then
 begin
  pp:=(dph and $7F)-64;           {pp in range -64 to +63}
  if pp=-64 then pp:=0           {pp now -63 to +63}
 end;
 if code=qpsk then
 begin
  pp:=(dph and $3F)-32;           {pp in range -32 to +31}
  if pp=-32 then pp:=0           {pp now -31 to +31}
 end;
 rxfreq:=rxfreq-pp*0.0002      {tweek rx freq towards centre a bit}
end;

procedure rcvdata(D,A:byte);
begin
 showscope(D,A);
 if not(simplex and txon) then decode(D);
    {decode unless simplex and sending}
 if dcd and afc
          then doafc(D)
end;

procedure settxfreq;        {show tx frequency same as rx frequency}
var S:string[20];
begin
 str(txfreq:6:1,S);
 S:='   '+S+'   ';
 setcolor(black);          {blank out the old one}
 mouseoff;
 if init then outtextXY(150,20,S);
 txfreq:=rxfreq;           {set the new one}
 str(txfreq:6:1,S);
 S:='TX='+S+' Hz';
 setcolor(white);
 outtextXY(150,20,S);       {show it}
 mouseon
end;

procedure txin(S:byte);
begin
 txbuf.data[txbuf.inptr]:=S;
 txbuf.inptr:=nextT(txbuf.inptr)
end;

procedure updatefreq(txf:boolean);
type cast=record
           low,mid,high,top:byte    {to get at the top byte in a longint}
          end;
var newfreq:string[6];
    S:string[12];
    rxflong:longint;
    I:integer;
begin
 str(rxfreq:6:1,newfreq);
 if not(txf) and (newfreq=rxfstr) then exit;    {same freq.. don't bother}
 if txbuf.inptr<>txbuf.outptr then exit;  {don't send unless empty buffer}
 S:='   '+rxfstr+'   ';
 setcolor(black);
 mouseoff;
 if init then OuttextXY(150,5,S);             {blank out old frequency}
 S:='RX='+newfreq+' Hz';
 setcolor(white);
 OuttextXY(150,5,S);                          {write in new frequency }
 mouseon;
 rxflong:=round(rxfreq/samplerate*256*65536); {form new freq as binary}
 for I:=0 to 3 do
 begin
  rxflong:=rxflong shl 6;                     {shift top 6 bits into top byte}
  txin(((cast(rxflong).top) and $3F)); {send it with push6 command}
 end;
 if txf then
 begin
  txin($83);
  settxfreq
 end else txin($81);
 rxfstr:=newfreq;                               {keep for next update}
end;

procedure setnet(N:boolean);
begin
 setcolor(black);
 mouseoff;
 if init then outtextXY(345,20,onoff[byte(neton)]);
 neton:=N;
 if neton then
  begin
   setcolor(lightgreen);
   updatefreq(true);
  end else setcolor(lightred);
 outtextXY(345,20,onoff[byte(neton)]);
 mouseon
end;

procedure setcode(C:Tcode);
const codestr:array[0..4] of string[12]=('','','F1: BPSK','F1: QPSK','F1: MORSE');
begin
 mouseoff;
 setcolor(black);
 if init then outtextXY(280,5,codestr[byte(code)]);
 code:=C;
 setcolor(white);
 outtextXY(280,5,codestr[byte(code)]);
 setcolor(lightblue);
  setlinestyle(solidln,0,3);
 circle(scopeX,scopeY,37);       {tuning scope surround}
 tuneptr:=0;
 setcolor(white);
 if code<>morse then
 begin
  line(scopeX-4,scopeY-36,scopeX+4,scopeY-36);
  line(scopeX-4,scopeY+38,scopeX+4,scopeY+38)      {top/bottom tuning marks}
 end;
 if code=qpsk then
 begin
  line(scopeX-37,scopeY-4,scopeX-37,scopeY+4);
  line(scopeX+36,scopeY-4,scopeX+36,scopeY+4)
 end;
 mouseon
end;

procedure sendcw(C:char);
const cwtxtab:array[32..93] of byte=       {ASCII to morsecode }
      ($80,$AC,$4A,$16,$00,$00,$00,$7A,$B4,$B6,$54,$14,$CE,$86,$56,$94,
       $FC,$7C,$3C,$1C,$0C,$04,$84,$C4,$E4,$F4,$00,$AA,$00,$8C,$44,$32,
       $00,$60,$88,$A8,$90,$40,$28,$D0,$08,$20,$78,$B0,$48,$E0,$A0,$F0,
       $68,$D8,$50,$10,$C0,$30,$18,$70,$98,$B8,$C8,$58,$E8,$6C);
var I:integer;
    symbol:byte;
begin
 C:=upcase(C);
 if not(C in [#32..#93]) then C:=' ';
 symbol:=cwtxtab[byte(C)];
 while symbol>0 do
 begin
  case symbol of
        $80: for I:=0 to 1 do txin($41);   {letter gap}
   $81..$FF: for I:=0 to 5 do txin($48);   {dash}
   $01..$7F: for I:=0 to 1 do txin($48);   {dot}
  end;
  for I:=0 to 1 do txin($41);   {element gap}
  symbol:=(symbol shl 1) and $FF;
 end
end;

procedure sendBchar(C:char);
var W:word;
begin
 if simplex then writescreen(C);
 W:=encotab[byte(C)];
 repeat
  txin($48+((W and 1) xor 1));
  W:=W shr 1
 until W<2
end;

procedure sendQchar(C:char);
var W:word;
    B:byte;
begin
 if simplex then writescreen(C);
 if rev then B:=$4E else B:=$4C;
 W:=encotab[byte(C)];
 repeat
  txin(B+((W and 1) xor 1));
  W:=W shr 1;
 until W<2
end;

procedure sendchar(C:char);
const txmode:Tcode=off;
var I:integer;
begin
 if C in [#0..#4] then
 begin
  if Tcode(C)<>txmode then
  case Tcode(C) of
   off: begin
         if txmode in [bpsk,qpsk] then for I:=0 to 2*dcdlimit do txin($48);
         txin($40);
         txon:=false;
         abort:=false
        end;
   tune: begin
          if neton then updatefreq(true);
          txin($48)
         end;
   bpsk,qpsk: begin
               if neton then updatefreq(true);
               if not(Txmode in [bpsk,qpsk]) then
                 for I:=0 to 2*dcdlimit do txin($49);
               txon:=true
              end;
   morse: begin
           if neton then updatefreq(true);
           if txmode in [bpsk,qpsk] then
           begin
            for I:=0 to 2*dcdlimit do txin($48);
            sendcw(' ');
            txon:=true
           end
          end
  end;
  txmode:=Tcode(C)
 end else
 begin
  case txmode of
   qpsk:    sendQchar(C);
   bpsk:    sendBchar(C);
   morse:   sendcw(C);
  end
 end
end;

procedure echo(C:char);
var I:integer;
begin
 I:=mouseread;
 if mouseY>400+65 then mouseoff;
 if C=#13 then C:=#27;
 if C=#10 then C:=#25;   {convert CR and LF to symbols}
 setcolor(black);
 outtextXY(145,65,echoline);                          {blank out old line}
 if C=#8 then echoline:=' '+copy(echoline,1,41) else
  echoline:=copy(echoline,2,41)+C;                    {add this character}
 setcolor(yellow);
 outtextXY(145,65,echoline);                           {show new line}
 mouseon
end;

procedure keybufin(C:char);
const txmode:Tcode=off;
begin
 if C>#127 then C:=char(keytab[byte(C)]);   {convert CP850 to ANSI}
 begin
  if Tcode(C) in [off,tune] then txmode:=Tcode(C) else
  if txmode<>code then
  begin
   keybuf.data[keybuf.inptr]:=char(code);
   keybuf.inptr:=nextK(keybuf.inptr);
   txmode:=code
  end;
  keybuf.data[keybuf.inptr]:=C;
  keybuf.inptr:=nextK(keybuf.inptr)
 end
end;

procedure cwident;
var I:integer;
    temp:Tcode;
begin
 temp:=code;
 code:=morse;
 keybufin(' ');
 for I:=1 to length(cwid) do keybufin(cwid[I]);
 keybufin(char(off));
 code:=temp
end;

procedure sendtraffic(C:char); far;     {called via keydo procedure variable}
var L:longint;
    data:byte;
begin
 echo(C);
 if (C=#10) and autolf then exit;
 if (C=' ') and (column>70) then
 begin
  keybufin(#13);
  if not(autolf) then keybufin(#10);
  column:=0;
  exit
 end;
 if C in [#32..#255] then inc(column);
 if C=#13 then column:=0;
 keybufin(C)
end;

procedure dofilename(C:char); far;  {called via keydo procedure variable}
begin
 setcolor(black);
 mouseoff;
 outtextXY(400,50,filename);         {blank out filename}
 case C of
  #13: begin                         {enter}
        keydo:=sendtraffic;          {switch back to traffic}
        outtextXY(350,50,'FILE:');
        if filename='' then exit;
        assign(outfile,filename);    {try and open the file}
        {$I-}
        reset(outfile);
        {$I+}
        if ioresult=0 then
        begin
         setfile(true);    {turn on file-send if opened OK}
         sendchar(char(code))
        end;
        mouseon;
        exit
       end;
  #33..#126: filename:=filename+upcase(C);   {filename character}
  #8: begin                                                {backspace}
       filename:=copy(filename,1,length(filename)-1)
      end;
  #27: begin                                         {escape}
        keydo:=sendtraffic;                  {switch back to traffic}
        outtextXY(350,50,'FILE:');           {blank out legend}
        mouseon;
        exit
       end;
 end;
 setcolor(white);
 outtextXY(400,50,filename);         {show filename so far}
 mouseon
end;

procedure sendfilechar;
var C:char;
begin
 if eof(outfile) then
 begin
  close(outfile);                 {shut it if finished it}
  setfile(false);
  exit
 end;
 read(outfile,C);
 column:=0;           {prevent auto-newline screwing things up}
 if C>#127 then C:=char(keytab[byte(C)]); {convert CP850 to ANSI}
 sendchar(C)
end;

procedure CQcall;
const kk:string[8]='pse K'#13#10;
var I,J:integer;
    CQstr:string;
begin
 CQstr:='CQ CQ CQ de '+callsign+' '+callsign+' '+callsign+#13#10;
 keybufin(#13);
 keybufin(#10);
 for J:=0 to 2 do
  for I:=1 to length(CQstr) do keybufin(CQstr[I]);
 for I:=1 to length(kk) do keybufin(kk[I]);
 CWident
end;

procedure transmit;
begin
 if Xonoff then
 begin
  if fileout then sendfilechar else
  if keybuf.inptr<>keybuf.outptr then
  begin
   sendchar(keybuf.data[keybuf.outptr]);
   keybuf.outptr:=nextK(keybuf.outptr)
  end
 end
end;

procedure receive;
var B:byte;
begin
 if rxbuf.inptr=rxbuf.outptr then exit;
 B:=rxbuf.data[rxbuf.outptr];
 rxbuf.outptr:=nextr(rxbuf.outptr);
 if B=fend then rxstate:=1 else if B=fesc then rxstate:=rxstate+128 else
 begin
  if rxstate>127 then
  begin
   if B=tfesc then B:=fesc;
   if B=tfend then B:=fend;
   rxstate:=rxstate and 127
  end;
  case rxstate of
   1: case B of
       0: rxstate:=2;     {demodulated data packet}
       1: rxstate:=4     {spectrum data packet}
       else rxstate:=0    {something else?}
      end;
   2: begin
       rxframe[0]:=B;  {save phase data}
       rxstate:=3
      end;
   3: begin
       rcvdata(rxframe[0],B and $1F);  {process phase/amplitude}
       Xonoff:=(B and $80)>0;      {set Xonoff state}
       overrange:=(B and $40)>0;      {set overrange state}
       transmit;
       rxstate:=0
      end;
   4..67: begin
           rxframe[rxstate-4]:=B;  {spectrum data}
           inc(rxstate);
           if rxstate=68 then
           begin
            showwaterfall;
            rxstate:=0
           end
          end
  end
 end
end;

procedure abortTx;
begin
 abort:=false;
 keybuf.inptr:=0;
 keybuf.outptr:=0;
 if fileout then
 begin
  close(outfile);
  setfile(false)
 end;
 sendchar(char(off))
end;

procedure dokey;
var C:char;
    I:integer;
begin
 if not(keypressed) then exit;
 C:=readkey;
 case C of
  #0: begin                 {process function/cursor keys}
       case readkey of
{F1}    #59: if code=bpsk then setcode(qpsk) else setcode(bpsk);
{F2}    #60: setdcd(not(dcd));
{F3}    #61: setafc(not(afc));
{F4}    #62: setnet(not(neton));
{F5}    #63: if abort then aborttx else
             begin
              keybufin(char(off));
              abort:=true
             end;
{F6}    #64: cwident;
{F7}    #65: CQcall;
{F8}    #66: setlog(not(logon));
{F9}    #67: begin
              setfile(false);
              keydo:=dofilename;
              setcolor(lightcyan);
              mouseoff;
              outtextXY(350,50,'FILE:');
              mouseon;
              filename:=''
             end;
{F10}   #68: quit:=true;
        #75: rxfreq:=rxfreq-1;        {left arrow}
        #77: rxfreq:=rxfreq+1;        {right arrow}
{altF6} #109: if code<>morse then setcode(morse);
       end
      end;
  #1..#255: if not(fileout) then keydo(C)
 end
end;

procedure domouse;
begin
 if clickleft then
 begin
  mouseoff;
  case mouseX of
   0..scopeX+34: setDCD(not(dcd));
   wfallX..wfallX+95: if (MouseY>wfallY+400) and (mouseY<wfallY+412) then
                       rxfreq:=rxfreq+(mouseX-wfallX-47.5)*5.20833;
   260..374: case mouseY of
              400..415: if code=bpsk then setcode(qpsk) else setcode(bpsk);
              416..431: setnet(not(neton));
              432..447: CQcall;
             end;
   375..479: case mouseY of
              400..415: setdcd(not(dcd));
              416..431: if abort then aborttx else
                        begin
                         keybufin(char(off));
                         abort:=true
                        end;
              432..447: setlog(not(logon));
             end;
   480..585: case mouseY of
              400..415: setafc(not(afc));
              416..431: cwident;
              432..447: begin
                         setfile(false);
                         keydo:=dofilename;
                         setcolor(lightcyan);
                         mouseoff;
                         outtextXY(350,50,'FILE:');
                         mouseon;
                         filename:=''
                        end;
              460..479: quit:=true;
             end
  end;
  mouseon
 end
end;

procedure comirq; interrupt;
var B:byte;
begin
 port[portadr[comportNr]+1]:=1;                {disable transmit irq}
 if (port[portadr[comportNr]+5] and 1)>0 then
 begin
  rxbuf.data[rxbuf.inptr]:=port[portadr[comportNr]];
  rxbuf.inptr:=nextr(rxbuf.inptr)
 end;
 if (port[portadr[comportNr]+5] and $20)>0 then   {if tx free then..}
 begin
  if txbuf.inptr<>txbuf.outptr then        {if there is something in txbuf..}
  begin
   port[portadr[comportNr]]:=txbuf.data[txbuf.outptr];     {send it}
   txbuf.outptr:=nextT(txbuf.outptr);                        {move past it}
   port[portadr[comportNr]+1]:=3               {enable tx irq when it's sent}
  end
 end;
 port[$20]:=$20                                            {clear the PIC}
end;

procedure initcom(baudrate:word);
var B:byte;
begin
 esc:=false;
 rxbuf.inptr:=0;
 rxbuf.outptr:=0;                                    {set buffers empty}
 txbuf.inptr:=0;
 txbuf.outptr:=0;
 keybuf.inptr:=0;
 keybuf.outptr:=0;
 port[portadr[comportNr]+3]:=$83;                  {8,n,1, access divider}
 portw[portadr[comportNr]]:=115200 div baudrate;   {set baudrate}
 port[portadr[comportNr]+3]:=$03;                  {8,n,1,access data}
 B:=port[portadr[comportNr]+5];
 B:=port[portadr[comportNr]];         {mop up any old data or errorflags}
 getintvec(intvec[comportNr],existcom);         {save pre-existing irq vector}
 setintvec(intvec[comportNr],@comirq);                {put ours in it's place}
 port[portadr[comportNr]+4]:=8;                       {enable irq hardware}
 port[$21]:=port[$21] and not(picmask[comportNr]);    {enable PIC input}
 port[portadr[comportNr]+1]:=1                        {enable rx interrupt}
end;

procedure closecom;
begin
 port[$21]:=port[$21] or picmask[comportNr];     {turn off PIC input}
 port[portadr[comportNr]+4]:=0;                  {disable irq}
 port[portadr[comportNr]+1]:=0;                  {interrupts off}
 setintvec(intvec[comportNr],existcom)     {put back the pre-existing irq vector}
end;

function caps(S:string):string;
var I:integer;
begin
 for I:=1 to length(S) do S[I]:=upcase(S[I]);
 caps:=S
end;

procedure doparam(P:byte);
var S:string;
    error:word;
begin
 S:=paramstr(P);
 if S[1] in ['/','-'] then
 case upcase(S[2]) of
  'C': callsign:=copy(S,3,255);
  'P': comportNr:=byte(S[3]) and 15;
  'B': boot:=true;
  'F': begin
        S:=copy(S,3,255);
        val(S,rxfreq,error);
        if error>0 then rxfreq:=1000
       end;
  'S': simplex:=true;
  'N': autoLF:=false;
  'R': rev:=true;
 end
end;

procedure doparams;
var parmNr:byte;
begin
 for parmNr:=1 to paramcount do doparam(parmNr)
end;

procedure setpsk31dir;
var N:namestr;
    E:extstr;
begin
 fsplit(paramstr(0),psk31dir,N,E)
end;

begin
 if pos('?',paramstr(1))>0 then
 begin
  writeln('31 baud BPSK/QPSK driver program for DSP56002EVM by G3PLX');
  writeln('Version 3.01  (with QPSK)');
  writeln;
  writeln('Available commandline parameters:');
  writeln('/c followed by your callsign  e.g. /cG3PLX');
  writeln('/p followed by your COM port  e.g. /p2 for COM2');
  writeln('/b to load PSK31.CLD with G3PLX BOOT ROM');
  writeln('/f followed by centre frequency   e.g.  /f800.0');
  writeln('/s for simplex (on tx: mute rx and show local copy)');
  writeln('/r to operate QPSK with an SSB radio in LOWER sideband');
  writeln('/n to disable auto-linefeed');
  writeln('Defaults: NoCall, COM1, no boot, 1000 Hz, duplex, USB, autoLF');
  writeln;
  halt
 end;
 comportNr:=1;
 callsign:='NoCall';
 boot:=false;
 init:=false;
{$ifdef IDE}
  psk31dir:='';
{$else}
  setpsk31dir;
{$endif}
 rxfreq:=1000;
 rxfstr:='';
 simplex:=false;
 autolf:=true;
 rev:=false;
 doparams;
 txfreq:=rxfreq;
 cwid:='de '+callsign;
 setevmport(comportNr);
 if boot then
 begin
  resetEVM;
  writeln('Loading PSK31C.CLD...');
  loadevm(psk31dir+'psk31c.cld',57600);  {load at 57600 baud}
  error:=evmresult;
  writeln(evmmsg(error));
  if error>0 then halt(1);
  runEvm(CLDstart);
  error:=evmresult;
  if error>0 then
  begin
   writeln(evmmsg(error));         {print out what was wrong}
   halt(1)
  end
 end;
 initvari(psk31dir+'alphabet.txt');
 if ioresult>0 then
 begin
  writeln('Unable to load varicode alphabet file');
  halt(1)
 end;
 initviterbi;       {initialise the viterbi decoder}
 quit:=false;
 Xonoff:=true;
 abort:=false;
 txon:=false;
 lasttx:=0;
 bitcount:=0;
 truecount:=0;
 falsecount:=0;
 startgraph;
 setviewport(0,400,639,479,clipoff);  {graphics below Y=400, bios text above}
 keydo:=sendtraffic;
 echoline:='                                          ';
 echo(' ');
 directvideo:=false;       {enable BIOS text on graphics screen}
 highvideo;
 setcolor(darkgray);
 rectangle(0,0,639,79);
 rectangle(140,60,483,75);      {border round tx box}
 for tuneptr:=0 to persist-1 do
 begin
  scope[tuneptr].X:=scopeX;       {initialise tuning display persistence history}
  scope[tuneptr].Y:=scopeY
 end;
 setcolor(white);
 rectangle(wfallX-1,wfallY-1,wfallX+96,wfallY+12);
 line(wfallX+48,wfallY-5,wfallX+48,wfallY+17);
 outtextXY(500,5,'F3: AFC');
 outtextXY(500,20,'F6: CW ident');
 outtextXY(280,20,'F4: NET');
 outtextXY(280,35,'F7: CQ call');
 outtextXY(390,5,'F2: DCD');
 outtextXY(390,20,'F5: TX off');
 outtextXY(390,35,'F8: LOG');
 outtextXY(500,35,'F9: File send');
 outtextXY(500,65,'F10: quit');
 setcode(bpsk);
 setfile(false);
 setlog(false);
 setafc(false);
 setdcd(false);
 column:=0;
 rxstate:=0;
 mousewindow(0,400,639,479);
 mouseon;
 initcom(19200);    {set up com port, enable com interrupt}
 setnet(true);
 settxfreq;
 init:=true;
 repeat
  receive;
  domouse;
  updatefreq(false);
  dokey
 until quit;
 if boot then resetEvm;
 restorecrtmode;
 closecom
end.