/*
 * PC driver program for PSK31.DSK.
 * Release 2 (18-Apr-1999)
 * by Andrew Senior, G0TJZ.
 * This software is for non-commerical, amateur radio use only.
 * Please see README.TXT for further details.
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <conio.h>
#include <time.h>
#include <graphics.h>
#include <dos.h>
#include <signal.h>

#undef debug

#define FALSE 0
#define TRUE !FALSE
#define FSAMP 8000

/* DSK loader commands */
#define LD 2
#define LP 3
#define XG 5

/* loaddsk() error codes */
#define NOFILE 1
#define BADFILE 2
#define NOENTRY 3
#define RESETFAIL 4
#define COMFAIL 5

/* Status display legend offsets */
#define TXLEGX 90
#define RXLEGX 440

/* Version string */
char verstext[] = "Release 2, 18-Apr-1999";

/* Serial port control data */
int combase = 0x3F8, comirq = 4, comintnr;
int picbase = 0x20, picmask = 0xFFFF;
int combases[4] = {0x3F8, 0x2F8, 0x3E8, 0x2E8};
int comirqs[4] = {4, 3, 4, 3};
void interrupt (*oldcomhandler)();
int dtrinvert = 0, rtsinvert = 0;
unsigned int loadbaud = 57600;

enum {fix, track, offset} txmode = track; /* Tx frequency control mode */
int quit = FALSE; /* Program exit flag */
double rxfreq = 1150, txfreq = 1150; /* Rx & Tx tone frequencies (Hz) */
double txoffs = 0; /* Tx-rx offset (Hz) */

/* Rx buffer control data */
#define RXBUFSIZE 256
struct rxdata {
	unsigned char pd; /* Phase difference */
	unsigned char amp; /* Amplitude (logarithmic) */
};
struct rxdata rxbuf[RXBUFSIZE]; /* Buffer for received data from DSK */
volatile int rxbufinptr = 0, bufcount = 0; /* Buffer control vars */
volatile int bufoverrun = FALSE;
int rxbufoutptr = 0;
#ifdef debug
	int rxbufhwm = 0;
#endif

/* Command buffer control data */
#define TXBUFSIZE 256
unsigned char txbuf[TXBUFSIZE];
volatile int txbufoutptr = 0, txbufcount = 0;
int txbufinptr = 0;

/* Receiver control data */
unsigned int rxreg; /* Rx shift register */
int vlerror = FALSE; /* Varicode error flag */
char decotab[2048];  /* Varicode <-> ASCII tables */
unsigned int encotab[256];
int usesquelch = 1; /* Squelch enabled flag */
double is = 0, qs = 0; /* Smoothed signal vectors for squelch/AFC */
double smindist = 0; /* Smoothed Viterbi minimum distance for QPSK squelch */
int showampl = 1; /* Show amplitude on tuning display flag */

#ifdef debug
	int proterrors = 0;
#endif

/* Transmitter control data */
#ifdef debug
	int txbufhwm = 0;
#endif
volatile int txsymbolready = FALSE;
volatile unsigned char txsymbolbuf;
int send = FALSE; /* Send mode flag */

/* Transmit buffer control data */
#define TEXTBUFSIZE 256
unsigned long textbuf[TEXTBUFSIZE];
int textbufoutptr = 0, textbufinptr = 0, textbufcount = 0;
/* Morse code encoding table */
unsigned int cwtab[] = {0x14, 0x4A, 0x00, 0x00, 0x00, 0x00, 0x7A, 0xB4,
												0xB6, 0x00, 0x54, 0xCE, 0x86, 0x56, 0x94, 0xFC,
												0x7C, 0x3C, 0x1C, 0x0C, 0x04, 0x84, 0xC4, 0xE4,
												0xF4, 0xE2, 0x00, 0x00, 0x00, 0x00, 0x32, 0x00,
												0x60, 0x88, 0xA8, 0x90, 0x40, 0x28, 0xD0, 0x08,
												0x20, 0x78, 0xB0, 0x48, 0xE0, 0xA0, 0xF0, 0x68,
												0xD8, 0x50, 0x10, 0xC0, 0x30, 0x18, 0x70, 0x98,
												0xB8, 0xC8};
char callsign[20] = "NOCALL"; /* User's callsign */
int qpsk = FALSE; /* QPSK mode flag */

/* Convolutional coder/Viterbi decoder data */
unsigned char symbols[32], poly1 = 0x19, poly2 = 0x17;
struct state {
	unsigned int dist;
	long estimate;
};
struct state states[16]; /* Decoder trellis */
int dcd = FALSE; /* DCD on flag */
int ptt = FALSE; /* PTT on flag */
int capture = FALSE; /* Capture to file flag */
FILE *capfp; /* Capture file stream pointer */
int filesend = FALSE; /* File send active flag */
FILE *sendfp; /* File send file stream pointer */
int afc = TRUE; /* AFC on flag */
/* Text window control structure */
struct winctrl {
	int tlx; /* X co-ord of top left corner */
	int tly; /* Y co-ord of top left corner */
	int brx; /* X co-ord of bottom right corner */
	int bry; /* Y co-ord of bottom right corner */
	int height; /* Window height */
	int x; /* Current X co-ord */
	int y; /* Current Y co-ord */
};
struct winctrl rxwin = {1, 1, 80, 22, 22, 1, 1}; /* Receive window */
struct winctrl txwin = {1, 24, 80, 25, 2, 1, 1}; /* Transmit window */
int lsb = FALSE; /* Reverse QPSK sense flag */
int simplex = FALSE; /* Simplex mode flag */
int keydown = 0; /* Transmitter audio tone on flag */
char *loaderrs[6] = {"No error",
										 "Load file does not exist",
										 "Unrecognised code in DSK file",
										 "No entry point specified in DSK file",
										 "Couldn't reset and initialise DSK",
										 "Communications failure with DSK"};
/* Code page 850 -> ANSI translation table */
unsigned char toansi[128] = {199, 252, 233, 226, 228, 224, 229, 231,
														 234, 235, 232, 239, 238, 236, 196, 197,
														 201, 230, 198, 244, 246, 242, 251, 249,
														 255, 214, 220, 248, 163, 216, 215, 131,
														 225, 237, 243, 250, 241, 209, 170, 186,
														 191, 174, 172, 189, 188, 161, 171, 187,
														 0,   0,   0,   0,   0,   193, 194, 192,
														 169, 0,   0,   0,   0,   162, 165, 0,
														 0,   0,   0,   0,   0,   0,   227, 195,
														 0,   0,   0,   0,   0,   0,   0,   164,
														 240, 208, 202, 203, 200, 0,   205, 206,
														 207, 0,   0,   0,   0,   166, 204, 0,
														 211, 223, 212, 210, 245, 213, 181, 254,
														 222, 218, 219, 217, 253, 221, 175, 180,
														 173, 177, 0,   190, 182, 167, 247, 184,
														 176, 168, 183, 185, 179, 178, 0,   160};
/* ANSI->code page 850 translation table */
unsigned char tocp850[128] = {0,   0,   0, 159,   0,   0,   0,   0,
															0,   0,   0,   0,   0,   0,   0,   0,
															0,   0,   0,   0,   0,   0,   0,   0,
															0,   0,   0,   0,   0,   0,   0,   0,
															255, 173, 189, 156, 207, 190, 221, 245,
															249, 184, 166, 174, 170, 240, 169, 238,
															248, 241, 253, 252, 239, 230, 244, 250,
															247, 251, 167, 175, 172, 171, 243, 168,
															183, 181, 182, 199, 142, 143, 146, 128,
															212, 144, 210, 211, 222, 214, 215, 216,
															209, 165, 227, 224, 226, 229, 153, 158,
															157, 235, 233, 234, 154, 237, 232, 225,
															133, 160, 131, 198, 132, 134, 145, 135,
															138, 130, 136, 137, 141, 161, 140, 139,
															208, 164, 149, 162, 147, 228, 148, 246,
															155, 151, 163, 150, 129, 236, 231, 152};

static void init_tables(void)
{
	FILE *fp;
	int k, n;
	unsigned int codeword, mask;
	char buf[14];

	fp = fopen("PSK31X.COD", "rt");
	if(fp == NULL) {
		printf("Can't open code table file PSK31.COD\n");
		exit(EXIT_FAILURE);
	}
	for(k = 0; k < 256; k++) {
		codeword = 4;	/* Preset 100 pattern */
		fgets(buf, 14, fp);
		for(n = strlen(buf) - 2; n >= 0; n--) {
			codeword <<= 1;
			if(buf[n] == '1')
				codeword++;
		}
		encotab[k] = codeword;
		mask = 0x8000;	/* Start at MSB */
		while((mask & codeword) == 0)
			mask >>= 1;	/* Find dummy '1' MSB added earlier */
		codeword &= ~mask;	/* Remove it */
		codeword >>= 1;	/* Remove (always '1') LSB */
		decotab[codeword] = k;
	}
	fclose(fp);
}

static void scroll(struct winctrl *wcp)
{
	union REGS r;

	r.x.ax = 0x0601;
	r.x.bx = 0x0000;
	r.h.ch = wcp->tly - 1;
	r.h.cl = wcp->tlx - 1;
	r.h.dh = wcp->bry - 1;
	r.h.dl = wcp->brx - 1;
	int86(0x10, &r, &r);
}

static void linefeed(struct winctrl *wcp)
{
	if(wherey() == wcp->height)
		scroll(wcp);
	else
		putch(10);
}

static void dispchar(char c, struct winctrl *wcp)
{
	window(wcp->tlx, wcp->tly, wcp->brx, wcp->bry);
	gotoxy(wcp->x, wcp->y);
	switch(c) {
		case 8:
			putch(8); /* Backspace */
			putch(' '); /* Erase old character */
			putch(8); /* Backspace again */
			break;
		case 7: /* Ignore bells */
			break;
		case 10:
			break; /* Ignore LFs */
		case 13:
			putch(13);
			linefeed(wcp); /* Always insert a LF */
			break;
		default:
			if(wherex() == 80) {
				putch(13);
				linefeed(wcp);
			}
			putch(c);
	}
	wcp->x = wherex();
	wcp->y = wherey();
}

static void xprint(char *text)
{
	int len, k;

	len = strlen(text);
	for(k = 0; k < len; k++)
		dispchar(text[k], &rxwin);
}

/* Send byte to DSK */

static void sendbyte(unsigned char u)
{
	while(txbufcount >= TXBUFSIZE - 1)
		; /* Wait for buffer to empty */
	txbuf[txbufinptr++] = u;
	txbufinptr %= TXBUFSIZE;
	if(++txbufcount >= TXBUFSIZE) {
		xprint("\rTx buffer overrun!\r");
		exit(1);
	}
	#ifdef debug
		if(txbufcount > txbufhwm)
			txbufhwm = txbufcount;
	#endif
}

void storeinrxbuf(unsigned char pd, unsigned char amp)
{
	rxbuf[rxbufinptr].pd = pd;
	rxbuf[rxbufinptr++].amp = amp;
	rxbufinptr %= RXBUFSIZE;
	if(++bufcount >= RXBUFSIZE) {
		bufoverrun = TRUE;
		bufcount = 1;
	}
	#ifdef debug
		if(bufcount > rxbufhwm)
			rxbufhwm = bufcount;
	#endif
}

static unsigned char encode_qpsk(unsigned char sreg)
{
	unsigned char symb;

	symb =  symbols[~sreg & 0x1F];
	if(lsb)
		symb = (4 - symb) & 3; /* Reverse 90/270 exchanges if on LSB */
	return symb;
}

static unsigned char encode_bpsk(unsigned char sreg)
{
	if(sreg & 1)
		return 0; /* 1 -> no change */
	else
		return 2; /* 0 -> reversal */
}

static unsigned char encode(unsigned char sreg, int useqpsk)
{
	if(useqpsk)
		return encode_qpsk(sreg);
	else
		return encode_bpsk(sreg);
}

static unsigned char getnextsymbol(void)
{
	static unsigned char shiftreg = 0; /* Transmit shift register */
	static unsigned long codewordbuf = 0; /* Current codeword buffer */
	static useqpsk = 0;
	unsigned char symb;

	symb = encode(shiftreg, useqpsk);
	if(keydown) {
		if(simplex)
			storeinrxbuf(symb << 6, 224); /* Loop back to rx if simplex mode */
		symb |= 4; /* Tone on */
	}
	shiftreg <<= 1;
	if(codewordbuf & 1)
		shiftreg |= 1;
	codewordbuf >>= 1;
	if(codewordbuf <= 1) {
		codewordbuf = 0;
		if(textbufcount > 0) {
			/* Get next codeword */
			codewordbuf = textbuf[textbufoutptr++];
			textbufoutptr %= TEXTBUFSIZE;
			textbufcount--;
			/* Read flag bits in LSBs of codeword */
			if(codewordbuf & 1)
				useqpsk = 1; /* Encode this one using QPSK */
			else
				useqpsk = 0; /* Encode using BPSK */
			if(codewordbuf & 2)
				keydown = 1; /* Tone should be on */
			else
				keydown = 0; /* Tone off */
			codewordbuf >>= 2; /* Shift out flag bits */
		}
	}
	return symb;
}

/*
 * This is the UART interrupt handler.
 * It's rather a mess, but fixes the bug in the beta release
 * which ended up with the UART apparently signalling an interrupt
 * but no interrupts actually occurring!
 * My theory is that the problem is down to the combination of the
 * UART's prioritised interrupts and the PIC chip being edge-triggered.
 * If both an RDA and a THRE interrupt are pending and you only service
 * the higher priority RDA, perhaps the IRQ line stays high (because the
 * THRE int is still pending), but the PIC chip doesn't see this as a
 * new interrupt?
 * Perhaps some expert might like to comment on this??? I'd be grateful.
 * Anyhow, thanks again to G3PLX as the ideas I've used below are
 * "borrowed" from his Pascal code again.
 */

void interrupt newcomisr(void)
{
	unsigned char rxbyte, txbyte;
	int wasrxint, bufwasntempty;
	static unsigned char dphase = 0, ampl = 0;

	wasrxint = bufwasntempty = 1;
	outportb(picbase, 0x20); /* Reset the PIC for this IRQ */
	if(comirq > 7)
		outportb(0x20, 0x20); /* Reset master PIC too */
	outportb(combase+1, 0x01); /* Disable tx interrupt */
	if(inportb(combase+2) == 0x04) {
		/* Rx data available */
		wasrxint = 1;
		rxbyte = inportb(combase); /* Get rx'd byte */
		switch(rxbyte & 3) {
			case 0: /* Top 6 bits of phase difference */
				dphase = rxbyte;
				break;
			case 1: /* Top 4 bits of amplitude + bottom 2 bits of p.d. */
				ampl = rxbyte & 0xF0;
				dphase |= (rxbyte >> 2) & 3;
				break;
			case 2: /* Bottom 4 bits of amplitude + reserved bits */
				ampl |= (rxbyte >> 2) & 0xF;
				if(!(simplex && keydown))
					storeinrxbuf(dphase, ampl); /* Process latest data */
				dphase = 0; /* Clear registers */
				ampl = 0;
				break;
			case 3:	/* Tx sync */
				#ifdef debug
					if(rxbyte != 3) {
						proterrors++; /* Protocol error detected */
						break;
					}
				#endif
				txbyte = getnextsymbol();
				txsymbolbuf = txbyte;
				txsymbolready = TRUE;
				break;
		}
	}
	if(inportb(combase+5) & 0x20) {
		/* UART tx is free */
		bufwasntempty = 0;
		if(txsymbolready) {
			/* Send next tx symbol */
			outportb(combase, txsymbolbuf);
			txsymbolready = FALSE;
			outportb(combase+1, 0x03); /* Enable tx interrupt */
		}
		else if(txbufcount > 0) {
			/* Send command data */
			outportb(combase, txbuf[txbufoutptr++]);
			txbufoutptr %= TXBUFSIZE;
			txbufcount--;
			outportb(combase+1, 0x03);
		}
	}
	/* Re-enable tx interrupt if tx still busy */
	if(wasrxint && bufwasntempty)
		outportb(combase+1, 0x03);
}

static int initcomhandler(void)
{
	/* Determine which PIC to use and which INT number */
	if(comirq < 8) {
		picbase = 0x20; /* Base address of PIC */
		comintnr = 0x8 + comirq; /* INT number for the IRQ */
		picmask = 1 << comirq; /* IRQ enable mask in PIC */
	}
	else {
		picbase = 0xA0;
		comintnr = 0x68 + comirq;
		picmask = 1 << (comirq - 8);
	}
	/* Check that the IRQ isn't already in use */
	if(!(inportb(picbase + 1) & picmask))
		return 1; /* IRQ already unmasked */
	oldcomhandler = getvect(comintnr); /* Save existing handler address */
	setvect(comintnr, newcomisr); /* Set new handler address */
	return 0; /* Success */
}

static int initcom(unsigned int baudrate, int nstop, int intflags)
{
	unsigned char ctl;

	if(intflags)
		if(initcomhandler())
			return 1;
	if(nstop == 2)
		ctl = 0x07; /* 2 stop bits */
	else
		ctl = 0x03; /* 1 stop bit */
	outportb(combase+3, ctl | 0x80); /* 8,N,nstop access divider */
	outport(combase, 115200 / baudrate); /* Set baudrate */
	outportb(combase+3, ctl); /* 8,N,nstop access data */
	if(intflags)
		ctl = 0x08; /* OUT2 on */
	else
		ctl = 0;
	if(!dtrinvert)
		ctl |= 0x01; /* DTR on */
	if(rtsinvert)
		ctl |= 0x02; /* RTS on */
	outportb(combase+4, ctl); /* Set control lines */
	inportb(combase+5); /* Read spurious indications... */
	inportb(combase);
	inportb(combase+6);
	outportb(combase+1, intflags); /* Set interrupt status */
	if(intflags)
		outportb(picbase + 1, inportb(picbase + 1) & ~picmask);
	return 0;
}

static void closecom(void)
{
	outportb(picbase + 1, inportb(picbase + 1) | picmask); /* Mask IRQ */
	outportb(combase + 1, 0); /* Disable serial interrupts */
	outportb(combase + 4, inportb(combase + 4) & 0xF7); /* Turn OUT2 off */
	setvect(comintnr, oldcomhandler);
}

/* Send byte directly (i.e. not through tx buffer) */

static void sendbytedirect(unsigned char byte)
{
	while(!(inportb(combase + 5) & 0x20))
		; /* Wait for THRE */
	outportb(combase, byte);
}

/* Send a byte to DSK and wait for echo in loader mode */

static int sendbytewait(unsigned char byte)
{
	unsigned char u;

	delay(1); /* Ensure DSK ready to receive */
	sendbytedirect(byte); /* Send it */
	while(!(inportb(combase + 5) & 0x01))
		; /* Wait for data ready */
	u = inportb(combase); /* Get the echoed byte */
	if(u != byte)
		return 0; /* Bytes don't match */
	return 1; /* OK */
}

/* Send word to DSK and wait for echo */

static int sendwordwait(unsigned int word)
{
	if(!sendbytewait(word / 256)) /* Send MS byte */
		return 0; /* Failed */
	if(!sendbytewait(word % 256)) /* Send LS byte */
		return 0; /* Failed */
	return 1; /* Success */
}

/* Reset DSK and set loader baud rate */

static int resetdsk(void)
{
	int tries;

	outportb(combase + 4, inportb(combase + 4) ^ 0x01); /* Reset DSK */
	delay(50); /* For 50 msec */
	outportb(combase + 4, inportb(combase + 4) ^ 0x01); /* Release reset */
	delay(200); /* Wait for DSK to boot */
	inportb(combase); /* Read any rubbish */
	for(tries = 10; tries > 0; tries--) {
		sendbytedirect(0x80); /* Send baudrate setting byte */
		delay(5); /* Allow for DSK to respond */
		if(inportb(combase) == 0x1B)
			break; /* ESC received = success */
	}
	if(tries <= 0)
		return 0; /* Failed to get response */
	return 1; /* OK */
}

/* Send load command */

static int startload(int space, unsigned int addr, unsigned int len)
{
	if(!sendbytewait(space)) /* Specify load type */
		return 0; /* Comms failure */
	if(!sendwordwait(addr)) /* Specify start address */
		return 0;
	if(!sendwordwait(len - 1)) /* Specify number of words */
		return 0;
	return 1; /* OK */
}

/* Load object file into DSK and run it */

int loaddsk(char *fname, unsigned int baudrate)
{
	FILE *fp;
	char line[128], buf[6], type;
	int done, ptr, count, space;
	unsigned int entry, start, data[32], words = 0;

	initcom(baudrate, 2, 0);
	printf("Resetting DSK... ");
	if(resetdsk())
		printf("OK\n");
	else {
		printf("Failed\n");
		return RESETFAIL;
	}

	fp = fopen(fname, "rt");
	if(fp == NULL)
		return NOFILE; /* Can't open object file */

	printf("Loading...\r");
	for(done = 0; !done;) {
		fgets(line, 128, fp);
		switch(line[0]) {
			case 0:
			case ':':
				done = 1; /* Finished loading */
				break;
			case '1':
				sscanf(&line[1], "%04x", &entry);	/* Entry point */
				break;
			case '9': /* Load data */
				switch(line[5]) {
					case 'M':
						space = LD; /* Data memory */
						break;
					case 'B':
						space = LP; /* Load program */
						break;
					case '7':
						space = 0; /* No data */
						break;
					default:
						fclose(fp);
						return BADFILE; /* Invalid type */
				}
				if(space != 0) {
					sscanf(&line[1], "%04x", &start); /* Start address of pgm or data */
					/* Read data words into array */
					count = 0;
					for(ptr = 5; ;ptr += 5) {
						sscanf(&line[ptr], "%c%04x", &type, &data[count]);
						if(type != '7')
							count++;
						else
							break; /* End of data */
					}
					if(!startload(space, start, count)) { /* Send load command */
						fclose(fp);
						return COMFAIL;
					}
					for(ptr = 0; ptr < count; ptr++) {
						if(!sendwordwait(data[ptr])) { /* Send data values */
							fclose(fp);
							return COMFAIL;
						}
					}
					words += count; /* Total word count */
					printf("Loading... %05d\r", words);
				}
				break;
			default: /* Ignore other lines in file */
				break;
		}
	}
	fclose(fp);
	/* Initialise IMR */
	if(!startload(LD, 4, 1))
		return COMFAIL;
	if(!sendwordwait(2)) /* Unmask INT2 */
		return COMFAIL;
	/* Start the program */
	if(entry != 0) {
		if(!sendbytewait(XG))
			return COMFAIL;
		if(!sendwordwait(entry))
			return COMFAIL;
		sendbytedirect(0); /* Start pulse - no reply from DSK */
	}
	else
		return NOENTRY; /* No entry point defined */
	printf("Loading... Done. \n");
	return 0;
}

static void showtxfreq(void)
{
	char buf[20];

	bar(223, 470, 223 + textwidth("8888.8 Hz"), 470 + textheight("8"));
	sprintf(buf, "%6.1f Hz", txfreq);
	setcolor(WHITE);
	outtextxy(223, 470, buf);
}

static void setdspfreq(double f)
{
	unsigned int w;

	w = (unsigned int)(f / (double)FSAMP * 65536 + 0.5);
	/* Send frequency data */
	sendbyte(((w & 0xF000) >> 12) | 0x40);
	sendbyte(((w & 0x0F00) >> 8) | 0x40);
	sendbyte(((w & 0x00F0) >> 4) | 0x40);
	sendbyte((w & 0x000F) | 0x40);
	/* Set this as the rx freq */
	sendbyte(0x80);
}

static void updatetxfreq(void)
{
	switch(txmode) {
		case track:
			/* Net tx freq to rx freq */
			sendbyte(0xC0); /* Net tx to rx command */
			txfreq = rxfreq;
			break;
		case offset:
			/* Keep constant tx-rx offset */
			txfreq = rxfreq + txoffs;
			setdspfreq(txfreq); /* Temporarily set rx to required tx freq */
			sendbyte(0xC0); /* Set tx freq = rx freq */
			setdspfreq(rxfreq); /* Set rx freq back again */
			break;
	}
	showtxfreq();
}

static void setfreq(double freq)
{
	unsigned int w;
	char buf[20];

	rxfreq = freq;
	setdspfreq(rxfreq);
	/* Update rx freq display */
	bar(348, 470, 348 + textwidth("8888.8 Hz"), 470 + textheight("8"));
	sprintf(buf, "%6.1f Hz", rxfreq);
	setcolor(WHITE);
	outtextxy(348, 470, buf);
	if(send)
		updatetxfreq();
}

static void process_rxchar(unsigned char c)
{
	if(c == 10)
		return; /* Ignore LFs */
	/* Translate ANSI to CP850 */
	if(c > 127)
		c = tocp850[c - 128];
	if(capture) {
		fputc(c, capfp);
		if(c == 13)
			fputc(10, capfp); /* Put LFs in */
	}
	dispchar(c, &rxwin);
}

static void decode_varicode(unsigned char bit)
{
	bit = !bit;	/* Invert received bit */
	rxreg >>= 1;
	/* Detect if codeword has gone past length limit (12 bits) */
	if((rxreg & 0x0002) > 0)
		vlerror = 1;
	/* Insert new bit */
	if(bit)
		rxreg |= 0x8000;
	/* If we detect the 00 pattern at top of shift reg AND
	 * the rest of the shift reg isn't all zero then we
	 * have a complete codeword - so process it!
	 */
	if(((rxreg & 0xC000) == 0) && (rxreg != 0)) {
		/* Shift right until the LSB is '1' */
		while((rxreg & 0x0001) == 0)
			rxreg >>= 1;
		/* Then boot out the '1'! */
		rxreg >>= 1;
		/* If the codeword isn't too long, is valid and DCD is on, print it */
		if(!vlerror && (rxreg <= 0x7FA) && dcd)
			process_rxchar(decotab[rxreg]);
		/* Reset error flag and shift reg for next codeword */
		vlerror = 0;
		rxreg = 0;
	}
}

/* The Viterbi decoder for QPSK.
 * Based on the Pascal code by G3PLX in PSK31 for the EVM
 * Now with soft decision
 */

static unsigned char soft_distance(unsigned char x, unsigned char y)
{
	unsigned char z;

	z = abs(x - y);
	if(z > 128)
		z = 256 - z;
	return z;
}

static unsigned char decode_qpsk_soft(unsigned char rcvd)
{
	unsigned int dists[32], min;
	long ests[32];
	unsigned char select, vote;
	int i;

	min = 65535; /* Preset minimum distance = huge */
	for(i = 0; i < 32; i++) {
		dists[i] = states[i / 2].dist + soft_distance(rcvd, symbols[i] * 64);
		if(dists[i] < min)
			min = dists[i];
		ests[i] = ((states[i / 2].estimate) << 1) + (i & 1);
	}
	/* Find smoothed minimum distance for QPSK squelch */
	smindist = 0.05*min + 0.95*smindist;
	for(i = 0; i < 16; i++) {
		if(dists[i] < dists[16 + i])
			select = 0;
		else
			select = 16;
		states[i].dist = dists[select + i] - min;
		states[i].estimate = ests[select + i];
	}
	vote = 0;
	for(i = 0; i < 16; i++)
		if(states[i].estimate & (1L << 20))
			vote++;
	if(vote == 8)
		return random(2);
	else
		return (unsigned char)(vote > 8);
}

static void doafc(char dp)
{
	static time_t lastsec = 0, nowsec;

	if(!dcd || ptt || !afc)
		return; /* Don't correct if: DCD is off, we're sending, or AFC disabled */
	if(qpsk)
		dp <<= 2; /* remove 2 MSBs */
	else
		dp <<= 1; /* remove 1 MSB */
	if(dp == -128) /* eliminate bias */
		dp = 0;
	rxfreq += dp * 0.0002; /* apply correction */
	/* Update frequency every second */
	time(&nowsec);
	if(nowsec != lastsec)
		setfreq(rxfreq);
	lastsec = nowsec;
}

static void dodcd(unsigned char pd)
{
	static unsigned long dcdshfreg;
	double isum, qsum, dp;
	int k, curdcd;

	curdcd = dcd; /* Record current DCD state */
	/* First do DCD based on reversals/carrier */
	dcdshfreg <<= 2; /* Shift register left */
	if(qpsk)
		dcdshfreg += ((pd + 0x20) & 0xC0) >> 6;
	else
		dcdshfreg += ((pd + 0x40) & 0x80) >> 6;
	if(dcdshfreg == 0xAAAAAAAAL) {
		dcd = TRUE;
		is = 1; qs = 0; /* Force smoothed vector to idle position */
		smindist = 0; /* Force smoothed minimum distance to zero */
	}
	else if(dcdshfreg == 0L) {
		dcd = FALSE;
		is = 0; qs = 0; /* Force smoothed vector to zero */
		smindist = 21; /* Force smoothed minimum distance over noise threshold */
	}
	else {
		/* The code-based DCD is inconclusive, so determine the
		 * DCD based on the signal quality
		 */
		if(qpsk) {
			/* For QPSK, signal quality is smoothed Viterbi min. dist. variable */
			if(smindist < 20)
				dcd = TRUE;
			else
				dcd = FALSE;
		}
		else {
			/* For BPSK, signal quality is modulus of smoothed vector */
			if(is*is + qs*qs > 0.25)
				dcd = TRUE;
			else
				dcd = FALSE;
		}
	}
	/* If we've turned the DCD on, do a one-off frequency correction */
	if(dcd && !curdcd && !ptt && afc) {
		/* Convert to smoothed diff phase in -pi...pi */
		if(is != 0 || qs != 0)
			dp = atan2(qs, is);
		else
			dp = 0;
		/* Convert to frequency error and apply correction */
		if(qpsk)
			dp *= 3.90625 / M_PI;
		else
			dp *= 7.8125 / M_PI;
		rxfreq += dp;
		setfreq(rxfreq);
	}
}

static void receive(void)
{
	static int pixel = 0, n;
	static struct {int x; int y;} persist[8];
	unsigned char diffphase, ampl;

	if(rxbufoutptr == rxbufinptr)
		return; /* Buffer empty */
	if(bufoverrun) {
		xprint(" Warning: Rx buffer overrun!\r");
		bufoverrun = FALSE;
		rxbufoutptr = rxbufinptr;
	}
	/* Get latest data out of rx buffer */
	ampl = rxbuf[rxbufoutptr].amp;
	diffphase = rxbuf[rxbufoutptr++].pd;
	rxbufoutptr %= RXBUFSIZE;
	bufcount--;
	/* "Fold" diffphase and find vector components for squelch */
	if(qpsk) n = 8;	else n = 4;
	is = 0.05*cos(n*M_PI*diffphase/256) + 0.95*is;
	qs = 0.05*sin(n*M_PI*diffphase/256) + 0.95*qs;
	/* Draw tuning display */
	if(!showampl)
		ampl = 224; /* Use full radius if not showing amplitude */
	setcolor(BLACK);
	line(320, 440, 320-persist[pixel].x, 440-persist[pixel].y);
	persist[pixel].x = (int)(0.156*ampl*sin(2*M_PI*diffphase/256)+0.5);
	persist[pixel].y = (int)(0.156*ampl*cos(2*M_PI*diffphase/256)+0.5);
	if(dcd)
		setcolor(GREEN);
	else
		setcolor(RED);
	line(320, 440, 320-persist[pixel].x, 440-persist[pixel].y);
	pixel = (pixel + 1) % 8;
	/* Do automatic frequency correction */
	doafc(diffphase);
	/* Do DCD */
	if(usesquelch)
		dodcd(diffphase);
	/* Decode the symbol */
	if(qpsk) {
		if(lsb)
			diffphase = (256 - diffphase) & 255;
		decode_varicode(decode_qpsk_soft(diffphase));
	}
	else {
		diffphase = ((diffphase + 0x40) & 0x80) >> 6;
		decode_varicode(diffphase);
	}
}

static void sendcodeword(unsigned long cword)
{
	textbuf[textbufinptr++] = cword;
	textbufinptr %= TEXTBUFSIZE;
	textbufcount++;
}

static void sendcodeword_wait(unsigned long cword)
{
	while(textbufcount == (TEXTBUFSIZE - 1))
		receive(); /* Keep rx going while waiting for buf to empty */
	sendcodeword(cword);
}

static void sendchar(unsigned char k)
{
	unsigned char c;
	unsigned long codeword;
	static int wrapneeded = FALSE;

	/* Translate CP850 to ANSI */
	if(k > 127)
		c = toansi[k - 128];
	else
		c = k;
	/* Return if code is NUL or can't be mapped to ANSI */
	if(c == 0)
		return;
	/* Do word wrap */
	if(!filesend) {
		if(txwin.x >= 70)
			wrapneeded = TRUE; /* Set wrap needed flag */
		if(wrapneeded) {
			if(c == 32) {
				c = '\r'; /* Replace space with CR if wrap needed */
				k = c;
				wrapneeded = FALSE; /* Clear flag */
			}
			else if(c == '\r')
				wrapneeded = FALSE; /* Clear flag since we've wrapped anyway */
		}
	}
	dispchar(k, &txwin); /* Display using CP850 code! */
	codeword = ((unsigned long)encotab[c] << 2) | 2; /* set tone on */
	if(qpsk)
		codeword |= 1; /* set use QPSK flag */
	sendcodeword(codeword);
}

static void showflag(int flag, int x, int y, char *true, char *false)
{
	int maxwidth;

	maxwidth = textwidth(false);
	if(textwidth(true) > maxwidth)
		maxwidth = textwidth(true);
	bar(x, y, x + maxwidth, y + textheight("A"));
	setcolor(WHITE);
	if(flag)
		outtextxy(x, y, true);
	else
		outtextxy(x, y, false);
}

static void showsquelch(void)
{
	showflag(usesquelch, RXLEGX, 430, " F9: Auto Squelch", " F9: Squelch off");
}

static void showmode(void)
{
	showflag(qpsk, TXLEGX, 440, "F4: QPSK", "F4: BPSK");
}

static void showtxmode(void)
{
	static char *txmlegs[3] = {"F1: Tx Fix", "F1: Tx Track", "F1: Tx Offset"};

	bar(TXLEGX, 410, TXLEGX + textwidth(txmlegs[offset]), 410 + textheight("A"));
	setcolor(WHITE);
	outtextxy(TXLEGX, 410, txmlegs[txmode]);
}

static void showafc(void)
{
	showflag(afc, RXLEGX, 440, "F10: AFC on", "F10: AFC off");
}

static void showcapture(void)
{
	showflag(capture, RXLEGX, 420, " F7: Capture on", " F7: Capture off");
}

static void setfilesend(void)
{
	char buf[72], *fname;
	int k;

	filesend = !filesend;
	if(filesend) {
		if(!send)
			return; /* Don't do it if not sending */
		/* Prompt for file name */
		window(1, 1, 80, 25);
		/* Clear bottom line of tx window */
		gotoxy(1, 25);
		for(k = 0; k < 76; k++)
			putch(' ');
		gotoxy(1, 25);
		cputs("File: ");
		buf[0] = 70; /* 70 characters max. */
		fname = cgets(buf);
		gotoxy(1, 25);
		for(k = 0; k < 76; k++)
			putch(' ');
		sendfp = fopen(fname, "rb");
		if(sendfp == NULL) {
    	/* If we can't open the file, ignore the request */
			filesend = FALSE;
			return;
		}
	}
	else
		/* Close the file when finished */
		fclose(sendfp);
}

static void sendpreamble(void)
{
	/* Sends 32 reversals at start of over */
	sendcodeword_wait(0x8002);
	sendcodeword_wait(0x8002);
	sendcodeword_wait(0x0102);
}

static void sendpostamble(void)
{
	/* Flushes shift register and sends 32 bits of carrier at end of over */
	if(qpsk)
		sendcodeword_wait(0x0083); /* Flush with 5 zeroes */
	sendcodeword_wait(0xFFFE); /* 32 bits of carrier */
	sendcodeword_wait(0xFFFE);
	sendcodeword_wait(0x01FE);
	sendcodeword_wait(0x0004); /* Tone off */
}

static void send_cw_char(int c)
{
	unsigned char u;
	int k;

	c = toupper(c);
	if(c < ' ' || c > 'Z')
		return;
	if(c == ' ')
		/* Inter-word space */
		sendcodeword_wait(0x0100); /* 6 bits key-up */
	else {
		u = cwtab[c - 33];
		while(u > 0) {
			if(u == 0x80) {
				/* 2 dots space:
				 * Combines with inter-element space to give
				 * inter-character space */
				sendcodeword_wait(0x0040); /* 4 bits key-up */
			}
			else if(u & 0x80) {
				/* 3 dots mark = 1 dash */
				sendcodeword_wait(0x01FE); /* 6 bits key-down */
			}
			else {
				/* 1 dot mark */
				sendcodeword_wait(0x001E); /* 2 bits key-down */
			}
			/* 1 dot inter-element space */
			sendcodeword_wait(0x0010); /* 2 bits key-up */
			u = u << 1;
		}
	}
}

static void showptt(void)
{
	showflag(ptt, 20, 420, "PTT ON", "     ");
}

static void ptton(void)
{
	unsigned char ctl;

	ptt = TRUE;
	ctl = inportb(combase+4);
	if(rtsinvert)
		ctl &= 0xFD; /* RTS off */
	else
		ctl |= 0x02; /* RTS on */
	outportb(combase+4, ctl); /* Turn PTT on */
	showptt();
	updatetxfreq(); /* Update tx freq (if needed) before sending */
}

static void pttoff(void)
{
	unsigned char ctl;

	ptt = FALSE;
	ctl = inportb(combase+4);
	if(rtsinvert)
		ctl |= 0x02; /* RTS on */
	else
		ctl &= 0xFD; /* RTS off */
	outportb(combase+4, ctl); /* Turn PTT off */
	showptt();
}

static void sendcwid()
{
	int k;

	if(send)
		sendpostamble(); /* If sending, send postamble first */
	else {
		/* If not sending, turn PTT on now */
		send = TRUE;
		ptton();
	}
	send_cw_char(' ');
	send_cw_char('D');
	send_cw_char('E');
	send_cw_char(' ');
	for(k = 0; k < strlen(callsign); k++)
		send_cw_char(callsign[k]);
	send = FALSE;
}

static void sendchar_wait(char c)
{
	while(textbufcount == (TEXTBUFSIZE - 1))
		receive();
	sendchar(c);
}

static void doautocq(void)
{
	static char cqmsg[] = "CQ CQ CQ de ";
	static char gamsg[] = "Pse K\r";
	int line, k, j;

	if(!send) {
		send = TRUE;
		ptton();
		sendpreamble();
	}
	sendchar_wait('\r');
	for(line = 0; line < 3; line++) {
		for(k = 0; k < strlen(cqmsg); k++)
			sendchar_wait(cqmsg[k]);
		for(j = 0; j < 3; j++) {
			for(k = 0; k < strlen(callsign); k++)
				sendchar_wait(callsign[k]);
			sendchar_wait(' ');
		}
		sendchar_wait('\r');
	}
	for(k = 0; k < strlen(gamsg); k++)
		sendchar_wait(gamsg[k]);
	sendcwid();
}

static void setfilecapture(void)
{
	capture = !capture;
	if(capture) {
		capfp = fopen("CAPTURE.TXT", "a+b");
		if(capfp == NULL)
			capture = FALSE;
	}
	else
		fclose(capfp);
	showcapture();
}

static void settxoffset(void)
{
	char buf[10];

	txmode = offset;
	showtxmode();
	window(txwin.tlx, txwin.tly, txwin.brx, txwin.bry);
	/* Blank the line */
	gotoxy(1, 2);
	cputs("                               ");
	/* Prompt for offset */
	gotoxy(1, 2);
	cputs("Enter tx offset (Hz): ");
	buf[0] = 8;
	cgets(buf);
	txoffs = atof(buf + 2);
	gotoxy(1, 2);
	cputs("                               ");
	/* Apply change if sending */
	if(send)
		updatetxfreq();
}

static void dokey(void)
{
	int k;

	if(!kbhit())
		return; /* No key pressed */
	k = getch();
	switch(k) {
		case 27:
			quit = TRUE;
			break;
		case 0:
			switch(getch()) {
				case 72: /* Up arrow */
					setfreq(rxfreq+1);
					break;
				case 141: /* Ctrl+up arrow */
					setfreq(rxfreq+10);
					break;
				case 80: /* Down arrow */
					setfreq(rxfreq-1);
					break;
				case 145: /* Ctrl+down arrow */
					setfreq(rxfreq-10);
					break;
				case 60: /* F2 */
					if(send) {
						send = FALSE;
						sendpostamble();
					}
					break;
				case 61: /* F3 */
					sendcwid();
					break;
				case 62: /* F4 */
					qpsk = !qpsk;
					showmode();
					break;
				case 67: /* F9 */
					usesquelch = !usesquelch;
					showsquelch();
					if(!usesquelch)
						dcd = 1;
					break;
				case 59: /* F1 */
					txmode = (txmode + 1) % 2; /* Toggle between fix/track */
					if(send)
						updatetxfreq(); /* Enforce changes now if sending */
					showtxmode();
					break;
				case 84: /* Shift+F1 */
					settxoffset();
					break;
				case 68: /* F10 */
					afc = !afc;
					showafc();
					break;
				case 65: /* F7 */
					setfilecapture();
					break;
				case 64: /* F6 */
					setfilesend();
					break;
				case 63: /* F5 */
					doautocq();
					break;
			}
			break;
		default:
			/* Traffic key */
			if(!send) {
				/* Start transmitting if not already */
				send = TRUE;
				ptton();
				sendpreamble();
			}
			if(textbufcount < (TEXTBUFSIZE - 1))
				sendchar(k);
			else {
				/* Warn user that buffer is full */
				sound(1000);
				delay(5);
				nosound();
			}
			break;
	}
}

static void dofilesend(void)
{
	char c;

	if(!filesend)
		return; /* File send not active */
	if(!send) {
		/* If no longer sending, abort the file send */
		setfilesend();
		return;
	}
	if(textbufcount < (TEXTBUFSIZE - 1)) {
		c = fgetc(sendfp);
		if(c == EOF) {
			setfilesend(); /* Stop sending */
			return;
		}
		sendchar(c);
	}
}

static int startgraph(void)
{
	int gd = VGA, gm = VGAHI;

	if(registerbgidriver(EGAVGA_driver) < 0)
		return 1; /* Couldn't register driver */
	initgraph(&gd, &gm, "");
	return 0;
}

/* Exit handler */

static void shutdown(void)
{
	int k;

	closegraph();
	closecom();
	if(capture)
		fclose(capfp);
	resetdsk();
	#ifdef debug
		printf("*** Debug information ***\n");
		printf("There were %d protocol errors.\n", proterrors);
		printf("Tx buffer high-water mark was %d.\n", txbufhwm);
		printf("Receive buffer high-water mark was %d.\n", rxbufhwm);
	#endif
}

/* Signal handler */

static void sighandler(int signum)
{
	shutdown();
	printf("\nThe program exited because a fatal error (type %d) occurred.\n", signum);
	printf("If the problem recurrs, please contact the author.\n");
	fflush(stdout);
	_exit(3);
}

static unsigned char parity(unsigned char u)
{
	unsigned char p, k;

	p = 0;
	for(k = 0; k < 8; k++) {
		p ^= u & 1;
		u >>= 1;
	}
	return p;
}

static void setport(int port)
{
	if(port < 1 || port > 4) {
		printf("Port number must be between 1 and 4\n");
		exit(EXIT_FAILURE);
	}
	combase = combases[port - 1];
	comirq = comirqs[port - 1];
}

static void checkrtsinvert(void)
{
	if(rtsinvert < 0 || rtsinvert > 1) {
		printf("RTSINVERT must be 0 or 1\n");
		exit(EXIT_FAILURE);
	}
}

static void checkdtrinvert(void)
{
	if(dtrinvert < 0 || dtrinvert > 1) {
		printf("DTRINVERT must be 0 or 1\n");
		exit(EXIT_FAILURE);
	}
}

static void checkloadbaud(void)
{
	if(loadbaud < 2400 || loadbaud > 57600) {
		printf("LOADBAUD must be between 2400 and 57600 baud inclusive.\n");
		exit(EXIT_FAILURE);
	}
}

static void checklsb(void)
{
	if(lsb < 0 || lsb > 1) {
		printf("LSB must be 0 or 1.\n");
		exit(EXIT_FAILURE);
	}
}

static void checksimplex(void)
{
	if(simplex < 0 || simplex > 1) {
		printf("SIMPLEX must be 0 or 1.\n");
		exit(EXIT_FAILURE);
	}
}

static void checkshowampl(void)
{
	if(showampl < 0 || showampl > 1) {
		printf("SHOWAMPL must be 0 or 1.\n");
		exit(EXIT_FAILURE);
	}
}

static void checkcall(void)
{
	if(strlen(callsign) > 19) {
		printf("CALL must be 19 characters or fewer.\n");
		exit(EXIT_FAILURE);
	}
}

static void readconfig(void)
{
	FILE *fp;
	char buf[40];
	float f;
	int port;

	fp = fopen("PSK31.INI", "rt");
	if(fp == NULL) {
		printf("Can't open configuration file PSK31.INI");
		exit(EXIT_FAILURE);
	}
	while(fgets(buf, 40, fp) != NULL) {
		strupr(buf);
		if(sscanf(buf, "PORT=%d", &port) == 1)
			setport(port);
		else if(sscanf(buf, "CALL=\"%[^\"]", callsign) == 1)
			checkcall();
		else if(sscanf(buf, "FREQ=%f", &f) == 1)
			txfreq = rxfreq = f;
		else if(sscanf(buf, "RTSINVERT=%d", &rtsinvert) == 1)
			checkrtsinvert();
		else if(sscanf(buf, "DTRINVERT=%d", &dtrinvert) == 1)
			checkdtrinvert();
		else if(sscanf(buf, "LOADBAUD=%u", &loadbaud) == 1)
			checkloadbaud();
		else if(sscanf(buf, "BASE=%x", &combase) == 1)
			printf("Serial port base address set to %Xh\n", combase);
		else if(sscanf(buf, "IRQ=%d", &comirq) == 1)
			printf("Serial port IRQ set to %d\n", comirq);
		else if(sscanf(buf, "LSB=%d", &lsb) == 1)
			checklsb();
		else if(sscanf(buf, "SIMPLEX=%d", &simplex) == 1)
			checksimplex();
		else if(sscanf(buf, "SHOWAMPL=%d", &showampl) == 1)
			checkshowampl();
	}
	fclose(fp);
}

static void initui(void)
{
	int k;

	setfillstyle(SOLID_FILL, BLACK);
	setfreq(rxfreq);
	sendbyte(0xC0); /* Make sure tx freq agrees with rx */
	showtxfreq();
	showmode();
	showtxmode();
	showafc();
	showcapture();
	showsquelch();
	outtextxy(TXLEGX, 430, "F3: Morse ID");
	outtextxy(TXLEGX, 450, "F5: Auto CQ");
	outtextxy(RXLEGX, 460, "Esc: exit");
	outtextxy(TXLEGX, 460, "F6: File send");
	outtextxy(TXLEGX, 420, "F2: Tx off");
	/* Banner between tx and rx windows */
	gotoxy(1, 23);
	for(k = 0; k < 31; k++)
		putch(0xCD);
	cputs(" Transmitted text ");
	for(k = 0; k < 31; k++)
		putch(0xCD);
	window(1, 1, 80, 22);
	setcolor(LIGHTGRAY);
	line(0, 400, 639, 400); /* Line between tx and status windows */
	/* Tuning display */
	setcolor(BROWN);
	setlinestyle(SOLID_LINE, 0, THICK_WIDTH);
	circle(320, 440, 38);
	/* Frequency display headings */
	setcolor(WHITE);
	outtextxy(375, 460, "Rx");
	outtextxy(250, 460, "Tx");
}

void main(void)
{
	int k;
	char buf[80];

	printf("\nPSK31 driver for TMS320C50 DSK\n%s\n\n", verstext);
	readconfig();
	directvideo = FALSE; /* BIOS text on graphics screen - thanks G3PLX */
	/* Generate convolutional coder symbols table */
	for(k = 0; k < 32; k++)
		symbols[k] = 2 * parity(k & poly1) + parity(k & poly2);
	/* Prime the Viterbi decoder */
	for(k = 0; k < 20; k++)
		decode_qpsk_soft(128);
	/* Load the DSK program */
	k = loaddsk("PSK31.DSK", loadbaud);
	if(k) {
		printf("DSK load failed: %s\n", loaderrs[k]);
		return;
	}
	delay(200); /* Make sure DSK is ready */
	init_tables(); /* Load Varicode tables */
	if(initcom(2000, 1, 0x01)) {
		printf("Couldn't install serial interrupt handler.\n");
		return;
	}
	if(startgraph()) {
		printf("Couldn't find graphics driver. PSK31.EXE may be corrupt.\n");
		closecom();
		resetdsk();
		return;
	}
	if(graphresult()) {
		printf("Couldn't initialise VGA graphics.\n");
		closecom();
		resetdsk();
		return;
	}
	if(atexit(shutdown)) {
		shutdown();
		printf("Couldn't install exit handler.\n");
		return;
	}
	/* Install signal handlers */
	signal(SIGABRT, sighandler);
	signal(SIGFPE, sighandler);
	signal(SIGILL, sighandler);
	signal(SIGINT, sighandler);
	signal(SIGSEGV, sighandler);
	signal(SIGTERM, sighandler);
	/* Draw user interface */
	initui();
	/* Main processing loop */
	while(!quit) {
		if(!send && ptt && (textbufcount == 0))
			pttoff();
		receive();
		dofilesend();
		dokey();
	}
	/* Exit handler will tidy up for us */
}