program qpsk31;           {psk31(evm) driver   QPSK version}
{

 Please contact the author peter.martinez@btinternet.com to report
 bugs or suggest changes. This is freeware for amateur use only

format of data from evm to PC:
PPPPPPPP: P (<>0) is 8-bit fractional receive phase.            At 31.25Hz
00000000: escape code: next byte as follows:
 00000000: receive phase = zero
 00000001: Sync byte: time to send another tx bit to EVM  At 31.25 Hz
 else ignored.
format of data from PC to evm:
00000TSS: carrier on when T=1; SS=phaseshift.
          SS: 00=carrier, 01=90 degrees HF,10=reversal, 11=90 degrees LF
          Should be sent in response to a sync byte from EVM
0001XXXX: not used
01DDDDDD: shift D into lsb-end of 24-bit register         When required
10XXXXXX: move 24-bit register to receive frequency       When required
11XXXXXX: move 24-bit register to transmit frequency      When required

}

uses dos,crt,graph,evm;
const portadr:array[1..4] of word=($3F8,$2F8,$3E8,$2E8);
      picmask:array[1..4] of byte=($10,$08,$10,$08);
      intvec:array[1..4] of byte=(12,11,12,11);
      tbufsize=10;      {bits from characters going to EVM}
      kbufsize=512;     {keypresses going to transmit characters}
      rbufsize=256;     {bits from EVM going to decoder}
      persist=8;        {tuning display persistence}
      samplerate=8000;   {evm sample rate for computing frequencies}
      dcdlimit=18;       {number of bits to test for carrier on/off detect}
      onoff:array[0..1] of string[3]=('OFF','ON');
      scopeX=74;
      scopeY=39;       {position of scope on screen}
      poly1=$19;
      poly2=$17;       {convolutional encoder polynomials}
      depth=20;        {Viterbi decoding delay (=4*k) }

type keyproc=procedure(C:char);
     Tcode=(bpsk,qpsk,morse);        {operating modes}
     Tplot=record                  {tuning display position}
      X,Y:word
     end;
     Tbuf=record                   {buffer structure}
           data:array[0..tbufsize-1] of byte;
           inptr,outptr:word
          end;
     Kbuf=record
           data:array[0..kbufsize-1] of byte;
           inptr,outptr:word
          end;
     Rbuf=record
           data:array[0..rbufsize-1] of byte;
           inptr,outptr:word
          end;
     Tstate=record        {a state in the trellis}
             dist:byte;     {total excess distance to this state}
             esti:longint;  {estimate of transmitted bit sequence}
            end;

var psk31dir:dirstr;
    callsign:string[10];
    cwid:string[20];
    comportNr:byte;
    error:byte;    {used to look at evm loader errors}
    rxbuf:Rbuf;
    txbuf:Tbuf;
    keybuf:Kbuf;    {buffers for stuff to/from evm}
    existcom:pointer; {com port interrupt vector to go back afterwards}
    varsreg:word;                     {for decoding varicode}
    rxdata,bitcount,column,truecount,falsecount:byte;
    init,boot,afc,neton,rev,autolf,soundon,fileout:boolean;
    simplex,logon,txon,quit,toolong,dcd,esc:boolean;
    rxfreq,txfreq:real;                    {frequencies in Hz}
    rxfstr:string[6];                      {rxfreq as shown on screen}
    outfile,logfile:text;
    filename:string[63];
    keydo:keyproc;
    code:Tcode;
    encotab:array[0..127] of word;      {encode table for varicode}
    decotab:array[0..511] of byte;      {decode table for varicode}
    echoline:string[42];                {echoes keyboard input}
    tune:array[0..persist-1] of Tplot;  {stores tuning display history}
    tuneptr:byte;                       {points to tuning display history}
    convshift:byte;                     {convolutional code shift register}
    lasttx:byte;                        {last transmit phase value}
    symbols:array[0..31] of byte;       {convolutional code symbols}
    trellis:array[0..15] of Tstate;     {viterbi decoder trellis}

{$L egavga.obj}
procedure egavga; external;     {dummy procedure for VGA driver}

procedure startgraph;
var GrDr,GrMo:integer;
begin
 GrDr:=registerbgidriver(@egavga);
 GrDr:= detect;
 initgraph(GrDr,GrMo,'');
 if graphresult<>0 then halt;
end;

function nextT(ptr:word):word;    {move buffer pointer around circular buffer}
begin
 nextT:=(ptr+1) mod Tbufsize
end;

function nextk(ptr:word):word;
begin
 nextk:=(ptr+1) mod kbufsize
end;

function nextr(ptr:word):word;
begin
 nextr:=(ptr+1) mod rbufsize
end;

procedure initvari(Fname:string);   {initialise varicode lookup tables}
var F:text;
    I:integer;
    W,mask:word;
    S:string[10];
begin
 assign(F,Fname);
 {$I-}
 reset(F);
 for I:=0 to 511 do decotab[I]:=240;    {prefill with error symbol}
 for I:=0 to 127 do
 begin
  readln(F,S);                          {get varicode string from file}
  W:=4;
  while S<>'' do                        {copy string chars into codeword bits}
  begin
   W:=W shl 1;
   if S[length(S)]='1' then inc(W);
   S:=copy(S,1,length(S)-1)
  end;
  encotab[I]:=W;                             {put codeword into encode table}
  mask:=$8000;
  while (mask and W)=0 do mask:=mask shr 1;  {find first non-zero bitposition}
  W:=(W and not(mask)) shr 1;                {zero it and shift out bit0}
  decotab[W]:=I                              {fill in decode table entry}
 end;
 close(F);
 varsreg:=0;
 toolong:=false
 {$I+}
end;

function parity(data:byte):byte;
var count:byte;
begin
 count:=0;
 while (data>0) do
 begin
  if odd(data) then inc(count);
  data:=data shr 1
 end;
 parity:=count and 1
end;

procedure setafc(A:boolean);
begin
 setcolor(black);
 if init then outtextXY(345,20,onoff[byte(afc)]);
 afc:=A;
 if afc then setcolor(lightgreen) else setcolor(lightred);
 outtextXY(345,20,onoff[byte(afc)]);
end;

procedure setnet(N:boolean);
begin
 setcolor(black);
 if init then outtextXY(565,5,onoff[byte(neton)]);
 neton:=N;
 if neton then
  begin
   setcolor(lightgreen);
   rxfstr:='' {force frequency update}
  end else setcolor(lightred);
 outtextXY(565,5,onoff[byte(neton)]);
end;

procedure setdcd(D:boolean);
begin
 setcolor(black);
 if init then outtextXY(455,5,onoff[byte(dcd)]);
 dcd:=D;
 if dcd then setcolor(lightgreen) else setcolor(lightred);
 outtextXY(455,5,onoff[byte(dcd)]);
end;

procedure setfile(S:boolean);
begin
 fileout:=S;
 if fileout then setcolor(lightgreen) else setcolor(black);
 outtextXY(350,50,'FILE SEND');
end;

procedure setlog(L:boolean);
begin
 setcolor(black);
 if init then outtextXY(455,35,onoff[byte(logon)]) else logon:=false;
 if L then
 begin
  assign(logfile,psk31dir+'PSK31.LOG');
  {$I-}
  append(logfile);
  {$I+}
  if ioresult>0 then rewrite(logfile)
 end else
 begin
  if logon then close(logfile)
 end;
 logon:=L;
 if logon then setcolor(lightgreen) else setcolor(lightred);
 outtextXY(455,35,onoff[byte(logon)])
end;

procedure setsound(S:boolean);
begin
 soundon:=S;
 if soundon then sound(round(rxfreq+0.5)) else nosound
end;

procedure scroll;
var R:registers;
begin
 R.ax:=$0601;     {scroll one "line" of 16 pixels}
 R.bx:=$0000;     {black}
 R.ch:=0;         {from top}
 R.cl:=0;         {from left}
 R.dh:=24;        {to bottom}
 R.dl:=79;        {to right}
 intr($10,R);     {get BIOS to do it}
end;

procedure linefeed;
begin
 if whereY=25 then scroll else write(#10)
end;

procedure writescreen(C:char);
begin
 if logon then
 begin
 {$I-}
  if (C<>#10) or not(autolf) then write(logfile,C);
  if (C=#13) and autolf then write(logfile,#10);
 {$I+}
  if ioresult>0 then setlog(false)
 end;
 case C of
   #8: write(#8' '#8);
  #10: if not(autolf) then linefeed;
  #13: begin
       write(#13);
       if autolf then linefeed;
      end;
   #32..#127,#240:
     begin
      if whereX=80 then
      begin
       write(#13);
       if autolf then linefeed;
      end;
      write(C)
     end
 end
end;

procedure writerx(B:byte);
begin
 if not(simplex and txon) then writescreen(char(B))
end;

procedure primetrellis; {set trellis as if it had been receiving 1's for ever}
var I:integer;
const dprime:array[0..15] of byte=(9,10,5,7,7,5,8,9,7,5,7,6,5,4,2,0);
begin
 for I:=0 to 15 do trellis[I].dist:=dprime[I];
 for I:=0 to 15 do trellis[I].esti:=$FFFFFFF0+I
end;

procedure dodcd(DP:byte);
var carrier,reversals:boolean;
    I:integer;
begin
 carrier:=(((DP+$20) shr 6) and 3)=0;
 reversals:=(((DP+$20) shr 6) and 3)=2;
 if carrier then
 begin
  truecount:=0;
  if falsecount<dcdlimit then
  begin
   inc(falsecount);
   if falsecount=dcdlimit then
   begin
    setdcd(false);
    varsreg:=0;              {prime decoders after dropout}
    toolong:=false;
    primetrellis;
   end
  end
 end else
 if reversals then
 begin
  falsecount:=0;
  if truecount<dcdlimit then
  begin
   inc(truecount);
   if truecount=dcdlimit then setdcd(true)
  end
 end else
 begin
  truecount:=0;
  falsecount:=0
 end
end;

procedure decodeB(bit:boolean);
begin
 varsreg:=varsreg shr 1;                     {shift previous bits}
 if varsreg and $0008>0 then toolong:=true;  {mark codes length>10}
 if bit then varsreg:=varsreg or $8000;      {add in this bit}
 if ((varsreg and $C000)=0) and (varsreg<>0) then      {if end of code}
 begin
  while (varsreg and $0001)=0 do varsreg:=varsreg shr 1;  {normalise to lsb}
  varsreg:=varsreg shr 1;                                 {shift out first 1}
  if toolong then writerx(240) else writerx(decotab[varsreg]);  {lookup and out}
  toolong:=false;                                            {reset flag}
  varsreg:=0                                                 {reset shiftreg}
 end
end;

function distance(A,B:byte):byte;         {distance=approx euclidean}
const dlookup:array[0..3,0..3] of byte=
      ((0,2,3,2),(2,0,2,3),(3,2,0,2),(2,3,2,0));
begin
 distance:=dlookup[A,B]
end;

procedure decodeV(rcvd:byte);    {The Viterbi Decoder}
var I,J:integer;
    dists:array[0..31] of byte;
    ests:array[0..31] of longint;
    select,min,vote:byte;
begin
 if rev then rcvd:=(4-rcvd) and 3;
 min:=255;
 for I:=0 to 31 do      {calc distances for all states and both data values}
 begin                 {added distance=distance between rcvd and predicted symbol}
  dists[I]:=trellis[I div 2].dist+distance(rcvd,symbols[I]);
  if dists[I]<min then min:=dists[I];    {keep track of the smallest distance}
  ests[I]:=((trellis[I div 2].esti) shl 1)+(I and 1)  {new estimate}
 end;
 for I:=0 to 15 do         {for each new state in the new trellis array}
 begin
  if dists[I]<dists[16+I] then select:=0 else select:=16;  {select lowest}
  trellis[I].dist:=dists[select+I]-min;          {update excess distances}
  trellis[I].esti:=ests[select+I]                {keep the new estimate}
 end;
 vote:=0;
 for I:=0 to 15 do if (trellis[I].esti and (1 shl depth))>0 then inc(vote);
 if vote=8 then decodeB(random(2)=1) else decodeB(vote<8);
end;

procedure initviterbi;
var I:integer;
    B:byte;
begin
 for I:=0 to 31 do                       {..generate the encode table}
  symbols[I]:=2*parity(I and poly1)+parity(I and poly2);
 primetrellis;
 convshift:=$FF             {initialise transmit shiftregister for idles}
end;

procedure decode(DP:byte);
begin
 dodcd(DP);
 if dcd then
 case code of
  qpsk:     decodeV(((DP+$20) shr 6) and 3);        {phase shift 0..3}
  bpsk: decodeB((DP+$40) and $FF<$80);              {reversal=logic 0}
 end
end;

procedure tunescope(dph:byte);
begin
 setcolor(black);
 line(tune[tuneptr].X,tune[tuneptr].Y,scopeX,scopeY);  {black out ancient line}
 tune[tuneptr].X:=scopeX-round(32*sin(pi*dph/128));
 tune[tuneptr].Y:=scopeY-round(32*cos(pi*dph/128))+1;  {calc coords of new line}
 if dcd then setcolor(yellow) else setcolor(red);
 line(tune[tuneptr].X,tune[tuneptr].Y,scopeX,scopeY);    {paint new line}
 tuneptr:=(tuneptr+1) mod persist               {count round persistence buffer}
end;

procedure doafc(dph:byte);
var pp:integer;
begin
 if code=morse then exit;
 if code=bpsk then
 begin
  pp:=(dph and $7F)-64;           {pp in range -64 to +63}
  if pp=-64 then pp:=0           {pp now -63 to +63}
 end;
 if code=qpsk then
 begin
  pp:=(dph and $3F)-32;           {pp in range -32 to +31}
  if pp=-32 then pp:=0           {pp now -31 to +31}
 end;
 rxfreq:=rxfreq-pp*0.0002      {tweek rx freq towards centre a bit}
end;

procedure receive;
var diffphase:byte;
begin
 if rxbuf.inptr=rxbuf.outptr then exit;          {nothing to do}
 diffphase:=rxbuf.data[rxbuf.outptr];            {get phaseshift}
 rxbuf.outptr:=nextR(rxbuf.outptr);
 decode(diffphase);
 tunescope(diffphase);
 if dcd and afc and (lasttx=0) then doafc(diffphase)
end;

procedure textin(data:byte);
begin
 while nextK(keybuf.inptr)=keybuf.outptr do receive;   {hang about if full}
 keybuf.data[keybuf.inptr]:=data;                      {put data into keybuf}
 keybuf.inptr:=nextK(keybuf.inptr);
 receive
end;

procedure sendevm(B:byte);
begin
 inline($FA);                           {mask interrupts temp}
 txbuf.data[txbuf.inptr]:=B;                     {put byte into the tx buffer}
 txbuf.inptr:=nextT(txbuf.inptr);                 {move past it}
 if (port[portadr[comportNr]+5] and $20)>0 then   {if tx free then..}
 begin
  port[portadr[comportNr]]:=txbuf.data[txbuf.outptr];   {send from txbuf}
  txbuf.outptr:=nextT(txbuf.outptr);                     {skip past it}
  port[portadr[comportNr]+1]:=3                   {enable tx irq after it}
 end;
 inline($FB)                          {re-enable interrupts}
end;

procedure settxfreq;        {show tx frequency same as rx frequency}
var S:string[20];
begin
 str(txfreq:6:1,S);
 S:='   '+S+'   ';
 setcolor(black);          {blank out the old one}
 if init then outtextXY(150,20,S);
 txfreq:=rxfreq;           {set the new one}
 str(txfreq:6:1,S);
 S:='TX='+S+' Hz';
 setcolor(white);
 outtextXY(150,20,S)       {show it}
end;

procedure updatefreq;
type cast=record
           low,mid,high,top:byte    {to get at the top byte in a longint}
          end;
var newfreq:string[6];
    S:string[12];
    rxflong:longint;
    I:integer;
begin
 str(rxfreq:6:1,newfreq);
 if newfreq=rxfstr then exit;              {same freq.. don't bother}
 if txbuf.inptr<>txbuf.outptr then exit;  {don't send unless empty buffer}
 S:='   '+rxfstr+'   ';
 setcolor(black);
 if init then OuttextXY(150,5,S);             {blank out old frequency}
 S:='RX='+newfreq+' Hz';
 setcolor(white);
 OuttextXY(150,5,S);                          {write in new frequency }
 rxflong:=round(rxfreq/samplerate*256*65536); {form new freq as binary}
 for I:=0 to 3 do
 begin
  rxflong:=rxflong shl 6;                     {shift top 6 bits into top byte}
  sendevm($40+((cast(rxflong).top) and $3F)); {send it with push4 command}
 end;
 sendevm($80);                                 {send 'update rxfreq' command}
 if neton then
 begin
  sendevm($C0);                                {send 'update txfreq' command}
  settxfreq                                    {paint in new tx freq}
 end;
 rxfstr:=newfreq;                               {keep for next update}
 setsound(soundon)                         {change sound pitch if required}
end;

procedure setptt(T:boolean);
var I,J:integer;
begin
 if fileout or (code=morse) then
 begin
  write(#7);    {don't mess with PTT while sending a file or on CW}
  exit
 end;
 setcolor(black);
 if init then outtextXY(450,20,onoff[byte(txon)]) else txon:=false;
 if T then setcolor(lightgreen) else setcolor(lightred);
 outtextXY(450,20,onoff[byte(T)]);
 if T=txon then exit;
 if T then J:=6 else J:=4;       {revs at turn-on: carrier at turn-off}
 for I:=0 to 2*dcdlimit do textin(J);
 if not(T) then textin(0);                {finish with txoff to kill tone}
 txon:=T;
end;

procedure setcode(C:Tcode);
const codestr:array[0..2] of string[12]=('F1: BPSK','F1: QPSK','F1: MORSE');
begin
 setcolor(black);
 if init then outtextXY(280,5,codestr[byte(code)]);
 if C=morse then setptt(false);
 code:=C;
 setcolor(white);
 outtextXY(280,5,codestr[byte(code)]);
end;

procedure sendcw(C:char);
const cwtxtab:array[32..93] of byte=       {ASCII to morsecode }
      ($80,$AC,$4A,$16,$00,$00,$00,$7A,$B4,$B6,$54,$14,$CE,$86,$56,$94,
       $FC,$7C,$3C,$1C,$0C,$04,$84,$C4,$E4,$F4,$00,$AA,$00,$8C,$44,$32,
       $00,$60,$88,$A8,$90,$40,$28,$D0,$08,$20,$78,$B0,$48,$E0,$A0,$F0,
       $68,$D8,$50,$10,$C0,$30,$18,$70,$98,$B8,$C8,$58,$E8,$6C);
var I:integer;
    symbol:byte;
begin
 C:=upcase(C);
 if not(C in [#32..#93]) then C:=' ';
 symbol:=cwtxtab[byte(C)];
 while symbol>0 do
 begin
  case symbol of
        $80: for I:=0 to 1 do textin(0);   {letter gap}
   $81..$FF: for I:=0 to 5 do textin(4);   {dash}
   $01..$7F: for I:=0 to 1 do textin(4);   {dot}
  end;
  for I:=0 to 1 do textin(0);   {element gap}
  symbol:=(symbol shl 1) and $FF;
 end
end;

procedure cwident;
var I:integer;
begin
 setPTT(false);
 for I:=1 to length(cwid) do sendcw(cwid[I])
end;

procedure sendvari(C:char);
var W:word;
begin
 W:=encotab[byte(C)];
 repeat
  if odd(W) then textin(4) else textin(6);
  W:=W shr 1
 until W<2
end;

procedure sendchar(C:char);
begin
 case code of
  qpsk:    sendvari(C);
  bpsk:    sendvari(C);
  morse:    sendcw(C);
 end
end;

procedure echo(C:char);
begin
 if simplex then writescreen(C);
 if C=#13 then C:=#27;
 if C=#10 then C:=#25;   {convert CR and LF to symbols}
 setcolor(black);
 outtextXY(145,65,echoline);                          {blank out old line}
 if C=#8 then echoline:=' '+copy(echoline,1,41) else
  echoline:=copy(echoline,2,41)+C;                    {add this character}
 setcolor(yellow);
 outtextXY(145,65,echoline)                           {show new line}
end;

procedure sendtraffic(C:char); far;     {called via keydo procedure variable}
var L:longint;
    data:byte;
begin
 if (code<>morse) and not(txon) then exit;     {don't transmit if tx is off}
 echo(C);
 if (C=#10) and autolf then exit;
 if (C=' ') and (column>70) then
 begin
  sendchar(#13);
  if not(autolf) then sendchar(#10);
  column:=0;
  exit;
 end;
 if C in [#32..#126] then inc(column);
 if C=#13 then column:=0;
 sendchar(C);
end;

procedure dofilename(C:char); far;  {called via keydo procedure variable}
begin
 setcolor(black);
 outtextXY(400,50,filename);         {blank out filename}
 case C of
  #13: begin                         {enter}
        keydo:=sendtraffic;          {switch back to traffic}
        outtextXY(350,50,'FILE:');
        if filename='' then exit;
        assign(outfile,filename);    {try and open the file}
        {$I-}
        reset(outfile);
        {$I+}
        if ioresult=0 then setfile(true);    {turn on file-send if opened OK}
        exit
       end;
  #33..#126: filename:=filename+upcase(C);   {filename character}
  #8: begin                                                {backspace}
       filename:=copy(filename,1,length(filename)-1)
      end;
  #27: begin                                         {escape}
        keydo:=sendtraffic;                  {switch back to traffic}
        outtextXY(350,50,'FILE:');           {blank out legend}
        exit
       end;
 end;
 setcolor(white);
 outtextXY(400,50,filename)         {show filename so far}
end;

procedure sendfilechar;
var C:char;
begin
 if eof(outfile) then
 begin
  close(outfile);                 {shut it if finished it}
  setfile(false);
  exit
 end;
 if keybuf.inptr<>keybuf.outptr then exit;  {don't fill the keyboard buffer}
 read(outfile,C);
 column:=0;           {prevent auto-newline screwing things up}
 sendtraffic(C)
end;

procedure CQcall;
const kk:string[8]='pse K'#13#10;
var I,J:integer;
    CQstr:string;
begin
 CQstr:='CQ CQ CQ de '+callsign+' '+callsign+' '+callsign+#13#10;
 if code<>morse then setPTT(true);                  {turn on the tx}
 sendtraffic(#13);
 sendtraffic(#10);
 for J:=0 to 2 do
 for I:=1 to length(CQstr) do
 begin                          {don't fill the keyboard buffer}
  repeat receive until (keybuf.inptr=keybuf.outptr) or keypressed;
  sendtraffic(CQstr[I]);
  if keypressed then exit
 end;
 for I:=1 to length(kk) do
  begin                          {don't fill the keyboard buffer}
  repeat receive until (keybuf.inptr=keybuf.outptr) or keypressed;
  sendtraffic(kk[I]);
  if keypressed then exit
 end;
 if code<>morse then CWident
end;

procedure dokey;
var C:char;
    I:integer;
begin
 if not(keypressed) then
 begin
  if fileout then sendfilechar;
  exit
 end;
 C:=readkey;
 case upcase(C) of
  #0: begin                 {process function/cursor keys}
       case readkey of
{F1}    #59: if code=bpsk then setcode(qpsk) else setcode(bpsk);
{F2}    #60: setdcd(not(dcd));
{F3}    #61: setnet(not(neton));
{F4}    #62: setafc(not(afc));
{F5}    #63: setptt(not(txon));
{F6}    #64: cwident;
{F7}    #65: setsound(not(soundon));
{F8}    #66: setlog(not(logon));
{F9}    #67: begin
              setfile(false);
              keydo:=dofilename;
              setcolor(lightcyan);
              outtextXY(350,50,'FILE:');
              filename:=''
             end;
{F10}   #68: quit:=true;
        #75: rxfreq:=rxfreq-1;        {left arrow}
        #115: rxfreq:=rxfreq-10;      {control left-arrow}
        #77: rxfreq:=rxfreq+1;        {right arrow}
        #116: rxfreq:=rxfreq+10;      {control right-arrow}
{altF1} #104: CQcall;
{altF6} #109: if code<>morse then setcode(morse);
       end
      end;
  #1..#127: if not(fileout) then keydo(C)
 end
end;

procedure comirq; interrupt;
var B:byte;
    sync:boolean;
begin
 sync:=false;
 port[portadr[comportNr]+1]:=1;                {disable transmit irq}
 if (port[portadr[comportNr]+5] and 1)>0 then
 begin
  B:=port[portadr[comportNr]];
  if esc then
  begin
   esc:=false;
   if B=1 then sync:=true;            {request for tx bit from EVM}
   if B=0 then
   begin
    rxbuf.data[rxbuf.inptr]:=B;               {transposed rxphase=0}
    rxbuf.inptr:=nextR(rxbuf.inptr)
   end
  end else
  begin
   if B=0 then esc:=true else                 {esc code rcvd}
   begin
    rxbuf.data[rxbuf.inptr]:=B;
    rxbuf.inptr:=nextR(rxbuf.inptr)                  {rx data rcvd}
   end
  end
 end;
 if sync then        {if EVM wants a tx bit..}
 begin
  if keybuf.inptr<>keybuf.outptr then   {if something to send..}
  begin
   B:=keybuf.data[keybuf.outptr];         {fetch it from the keyboard buffer}
   keybuf.outptr:=nextK(keybuf.outptr);
   lasttx:=B
  end else B:=lasttx;                   {..else repeat last bit}
  if code=qpsk then
  begin
   convshift:=((convshift shl 1) and $FF)+((B shr 1) and 1); {shift it into the convoluter}
   B:=(B and 4)+symbols[convshift and 31];           {look up the symbol}
   if rev and (B in [5,7]) then B:=B xor 2
  end;
  txbuf.data[txbuf.inptr]:=B;                 {put it in the txbuf}
  txbuf.inptr:=nextT(txbuf.inptr);             {move past it}
 end;
 if (port[portadr[comportNr]+5] and $20)>0 then   {if tx free then..}
 begin
  if txbuf.inptr<>txbuf.outptr then        {if there is something in txbuf..}
  begin
   port[portadr[comportNr]]:=txbuf.data[txbuf.outptr];     {send it}
   txbuf.outptr:=nextT(txbuf.outptr);                        {move past it}
   port[portadr[comportNr]+1]:=3               {enable tx irq when it's sent}
  end
 end;
 port[$20]:=$20                                            {clear the PIC}
end;

procedure initcom(baudrate:word);
var B:byte;
begin
 esc:=false;
 rxbuf.inptr:=0;
 rxbuf.outptr:=0;                                    {set buffers empty}
 txbuf.inptr:=0;
 txbuf.outptr:=0;
 keybuf.inptr:=0;
 keybuf.outptr:=0;
 port[portadr[comportNr]+3]:=$83;                  {8,n,1, access divider}
 portw[portadr[comportNr]]:=115200 div baudrate;   {set baudrate}
 port[portadr[comportNr]+3]:=$03;                  {8,n,1,access data}
 B:=port[portadr[comportNr]+5];
 B:=port[portadr[comportNr]];         {mop up any old data or errorflags}
 getintvec(intvec[comportNr],existcom);         {save pre-existing irq vector}
 setintvec(intvec[comportNr],@comirq);                {put ours in it's place}
 port[portadr[comportNr]+4]:=8;                       {enable irq hardware}
 port[$21]:=port[$21] and not(picmask[comportNr]);    {enable PIC input}
 port[portadr[comportNr]+1]:=1                        {enable rx interrupt}
end;

procedure closecom;
begin
 port[$21]:=port[$21] or picmask[comportNr];     {turn off PIC input}
 port[portadr[comportNr]+4]:=0;                  {disable irq}
 port[portadr[comportNr]+1]:=0;                  {interrupts off}
 setintvec(intvec[comportNr],existcom)     {put back the pre-existing irq vector}
end;

function caps(S:string):string;
var I:integer;
begin
 for I:=1 to length(S) do S[I]:=upcase(S[I]);
 caps:=S
end;

procedure doparam(P:byte);
var S:string;
    error:word;
begin
 S:=paramstr(P);
 if S[1] in ['/','-'] then
 case upcase(S[2]) of
  'C': callsign:=copy(S,3,255);
  'P': comportNr:=byte(S[3]) and 15;
  'B': boot:=true;
  'F': begin
        S:=copy(S,3,255);
        val(S,rxfreq,error);
        if error>0 then rxfreq:=1000
       end;
  'S': simplex:=true;
  'N': autoLF:=false;
  'R': rev:=true;
 end
end;

procedure doparams;
var parmNr:byte;
begin
 for parmNr:=1 to paramcount do doparam(parmNr)
end;

procedure setpsk31dir;
var N:namestr;
    E:extstr;
begin
 fsplit(paramstr(0),psk31dir,N,E)
end;

begin
 if pos('?',paramstr(1))>0 then
 begin
  writeln('31 baud BPSK/QPSK driver program for DSP56002EVM by G3PLX');
  writeln('Version 3.01  (with QPSK)');
  writeln;
  writeln('Available commandline parameters:');
  writeln('/c followed by your callsign  e.g. /cG3PLX');
  writeln('/p followed by your COM port  e.g. /p2 for COM2');
  writeln('/b to load PSK31.CLD with G3PLX BOOT ROM');
  writeln('/f followed by centre frequency   e.g.  /f800.0');
  writeln('/s for simplex (on tx: mute rx and show local copy)');
  writeln('/r to operate QPSK with an SSB radio in LOWER sideband');
  writeln('/n to disable auto-linefeed');
  writeln('Defaults: NoCall, COM1, no boot, 1000 Hz, duplex, USB, autoLF');
  writeln;
  halt
 end;
 comportNr:=1;
 callsign:='NoCall';
 boot:=false;
 init:=false;
{$ifdef IDE}
  psk31dir:='';
{$else}
  setpsk31dir;
{$endif}
 rxfreq:=1000;
 rxfstr:='';
 simplex:=false;
 autolf:=true;
 rev:=false;
 doparams;
 txfreq:=rxfreq;
 cwid:='de '+callsign;
 setevmport(comportNr);
 if boot then
 begin
  resetEVM;
  writeln('Loading PSK31.CLD...');
  loadevm(psk31dir+'psk31.cld',57600);  {load at 57600 baud}
  error:=evmresult;
  writeln(evmmsg(error));
  if error>0 then halt(1);
  runEvm(CLDstart);
  error:=evmresult;
  if error>0 then
  begin
   writeln(evmmsg(error));         {print out what was wrong}
   halt(1)
  end
 end;
 initvari(psk31dir+'alphabet.dat');
 if ioresult>0 then
 begin
  writeln('Unable to load varicode alphabet file');
  halt(1)
 end;
 initviterbi;       {initialise the viterbi decoder}
 quit:=false;
 soundon:=false;
 lasttx:=0;
 bitcount:=0;
 truecount:=0;
 falsecount:=0;
 startgraph;
 setviewport(0,400,639,479,clipoff);  {graphics below Y=400, bios text above}
 keydo:=sendtraffic;
 echoline:='                                          ';
 echo(' ');
 directvideo:=false;       {enable BIOS text on graphics screen}
 highvideo;
 setcolor(darkgray);
 rectangle(0,0,639,79);
 rectangle(140,60,483,75);      {border round control panel and tx box}
 setcolor(lightblue);
 setlinestyle(solidln,0,3);
 circle(scopeX,scopeY,37);       {tuning scope surround}
 for tuneptr:=0 to persist-1 do
 begin
  tune[tuneptr].X:=scopeX;       {initialise tuning display persistence history}
  tune[tuneptr].Y:=scopeY
 end;
 tuneptr:=0;
 setcolor(white);
 line(scopeX-4,scopeY-36,scopeX+4,scopeY-36);
 line(scopeX-4,scopeY+38,scopeX+4,scopeY+38);     {top/bottom tuning marks}
 outtextXY(500,5,'F3: NET');
 outtextXY(500,20,'F6: cwid');
 outtextXY(280,20,'F4: AFC');
 outtextXY(280,35,'F7: sound');
 outtextXY(390,5,'F2: DCD');
 outtextXY(390,20,'F5: TX');
 outtextXY(390,35,'F8: LOG');
 outtextXY(500,35,'F9: file');
 outtextXY(500,65,'F10: quit');
 setcode(bpsk);
 setfile(false);
 setlog(false);
 setafc(false);
 setnet(true);
 setdcd(false);
 setptt(false);
 column:=0;
 init:=true;
 initcom(19200);    {set up com port, enable com interrupt}
 repeat
  updatefreq;
  receive;
  dokey
 until quit;
 if boot then resetEvm;
 nosound;
 restorecrtmode;
 closecom
end.